<?php

namespace App\Controllers;

use App\Models\UserModel;
use App\Models\RoleModel;
use App\Libraries\LogAktivitas;

class User extends BaseController
{
    protected $userModel;
    protected $roleModel;
    protected $logAktivitas;

    public function __construct()
    {
        parent::initController(
            service('request'),
            service('response'),
            service('logger')
        );
        helper(['form', 'url']);
        $this->userModel = new UserModel();
        $this->roleModel = new RoleModel();
        $this->logAktivitas = new LogAktivitas();
    }

    public function index()
    {
        $permissionCheck = $this->checkPermission('user_view');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $users = $this->userModel->findAll();

        return view('user/index', [
            'users' => $users,
            'menus' => $this->data['menus'],
            'user'  => $this->data['user']
        ]);
    }

    public function create()
    {
        $permissionCheck = $this->checkPermission('user_create');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data = [
            'roles' => $this->roleModel->findAll()
        ];

        return view('user/create', array_merge($this->data, $data));
    }

    public function store()
    {
        $permissionCheck = $this->checkPermission('user_create');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $rules = [
            'name'     => 'required',
            'email'    => 'required|valid_email|is_unique[users.email]',
            'password' => 'required|min_length[6]',
            'role_id'  => 'required|integer',
        ];

        if (!$this->validate($rules)) {
            return view('user/create', array_merge($this->data, [
                'validation' => $this->validator,
                'roles'      => $this->roleModel->findAll()
            ]));
        }

        $data = [
            'name'     => $this->request->getPost('name'),
            'email'    => $this->request->getPost('email'),
            'password' => password_hash($this->request->getPost('password'), PASSWORD_DEFAULT),
            'role_id'  => $this->request->getPost('role_id'),
        ];

        $this->userModel->save($data);
        $userId = $this->userModel->getInsertID();
        $user = $this->userModel->find($userId);

        // Log user creation
        $this->logAktivitas->log('create_user', $this->data['user']['id'], $this->data['user']['name'], current_url());

        session()->setFlashdata('success', 'User created successfully');
        return redirect()->to('/users');
    }

    public function edit($id = null)
    {
        $permissionCheck = $this->checkPermission('user_edit');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $user = $this->userModel->find($id);
        if (!$user) {
            throw \CodeIgniter\Exceptions\PageNotFoundException::forPageNotFound('User not found');
        }

        $data = [
            'user'  => $user,
            'roles' => $this->roleModel->findAll()
        ];

        return view('user/edit', array_merge($this->data, $data));
    }

    public function update($id = null)
    {
        $permissionCheck = $this->checkPermission('user_edit');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $user = $this->userModel->find($id);
        if (!$user) {
            throw \CodeIgniter\Exceptions\PageNotFoundException::forPageNotFound('User not found');
        }

        $rules = [
            'name'    => 'required',
            'email'   => "required|valid_email|is_unique[users.email,id,{$id}]",
            'role_id' => 'required|integer',
        ];

        if (!$this->validate($rules)) {
            return view('user/edit', array_merge($this->data, [
                'validation' => $this->validator,
                'user'       => $user,
                'roles'      => $this->roleModel->findAll()
            ]));
        }

        $data = [
            'id'      => $id,
            'name'    => $this->request->getPost('name'),
            'email'   => $this->request->getPost('email'),
            'role_id' => $this->request->getPost('role_id'),
        ];

        if ($password = $this->request->getPost('password')) {
            $data['password'] = password_hash($password, PASSWORD_DEFAULT);
        }

        $this->userModel->save($data);

        // Log user update
        $this->logAktivitas->log('update_user', $this->data['user']['id'], $this->data['user']['name'], current_url());

        // Clear user permissions cache to refresh from role
        if (session()->has('user_permissions')) {
            session()->remove('user_permissions');
        }

        session()->setFlashdata('success', 'User updated successfully');
        return redirect()->to('/users');
    }

    public function delete($id = null)
    {
        $permissionCheck = $this->checkPermission('user_delete');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        if (!$this->userModel->find($id)) {
            throw \CodeIgniter\Exceptions\PageNotFoundException::forPageNotFound('User not found');
        }

        $this->userModel->delete($id);

        // Log user deletion
        $this->logAktivitas->log('delete_user', $this->data['user']['id'], $this->data['user']['name'], current_url());

        session()->setFlashdata('success', 'User deleted successfully');
        return redirect()->to('/users');
    }
}
