<?php

namespace App\Controllers;

use App\Models\Unit_kerjaModel;
use App\Models\DosenModel;
use CodeIgniter\Controller;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\IOFactory;
use App\Libraries\LogAktivitas;


class Unit_kerjaController extends BaseController
{
    protected $Unit_kerjaModel;
    protected $DosenModel;
    protected $logAktivitas;


    public function __construct()
    {
        $this->Unit_kerjaModel = new Unit_kerjaModel();
        $this->DosenModel = new DosenModel();
        $this->logAktivitas = new LogAktivitas();
    }

    private function getUser()
    {
        return [
            'id'   => session()->get('id'),
            'name' => session()->get('name'),
        ];
    }

    public function index()
    {
        $session = session();
        if (!$session->get('isLoggedIn')) {
            return redirect()->to('/login');
        }

        $permission = $this->checkPermission('unit_kerja_view');
        if ($permission instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permission;
        }

        $data['items'] = $this->Unit_kerjaModel
            ->select('unit_kerja.*, dosen.nama_lengkap')
            ->join('dosen', 'dosen.id_dosen = unit_kerja.id_dosen', 'left')
            ->findAll();

        $user = $this->getUser();
        $this->logAktivitas->log(
            'akses_unit_kerja',
            $user['id'],
            $user['name'],
            current_url()
        );

        return view('unit_kerja/index', $this->data + $data);
    }


    public function create()
    {
        $permission = $this->checkPermission('unit_kerja_create');
        if ($permission instanceof \CodeIgniter\HTTP\RedirectResponse) return $permission;

        $data['dosen'] = $this->DosenModel->findAll(); // Untuk dropdown

        return view('unit_kerja/create', $this->data + $data);
    }

    public function store()
    {
        $permission = $this->checkPermission('unit_kerja_create');
        if ($permission instanceof \CodeIgniter\HTTP\RedirectResponse) return $permission;

        $user = $this->getUser();

        $data = [
            'nama_unit'  => $this->request->getPost('nama_unit'),
            'id_dosen'   => $this->request->getPost('id_dosen'),
            'created_by' => $user['id'],
        ];

        $this->Unit_kerjaModel->insert($data);
        $idBaru = $this->Unit_kerjaModel->getInsertID();

        // 🔥 ambil dosen pakai id_dosen
        $dosen = $this->DosenModel
            ->where('id_dosen', $data['id_dosen'])
            ->first();

        $keterangan = [
            '<strong>Data yang ditambahkan:</strong>',
            '<strong>ID</strong>: ' . $idBaru,
            '<strong>Nama Unit</strong>: ' . esc($data['nama_unit']),
            '<strong>Dosen</strong>: ' . esc($dosen['nama_lengkap'] ?? '-'),
        ];

        $this->logAktivitas->log(
            'create_unit_kerja',
            $user['id'],
            $user['name'],
            current_url(),
            implode('<br>', $keterangan)
        );

        return redirect()->to('/unit_kerja')->with('success', 'Unit kerja berhasil ditambahkan.');
    }





    public function edit($id)
    {
        $permission = $this->checkPermission('unit_kerja_edit');
        if ($permission instanceof \CodeIgniter\HTTP\RedirectResponse) return $permission;

        $data['item']  = $this->Unit_kerjaModel->find($id);
        $data['dosen'] = $this->DosenModel->findAll();

        $user = $this->getUser();
        $this->logAktivitas->log(
            'edit_unit_kerja',
            $user['id'],
            $user['name'],
            current_url()
        );

        return view('unit_kerja/edit', $this->data + $data);
    }



    public function update($id)
    {
        $permissionCheck = $this->checkPermission('unit_kerja_edit');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        // 🔴 DATA LAMA
        $lama = $this->Unit_kerjaModel->find($id);
        if (!$lama) {
            return redirect()->to('/unit_kerja')->with('error', 'Data tidak ditemukan');
        }

        $post = $this->request->getPost();
        $perubahan = [];

        // 🔵 RESOLVE DOSEN LAMA & BARU
        $dosenLama = $this->DosenModel->find($lama['id_dosen']);
        $dosenBaru = $this->DosenModel->find($post['id_dosen']);

        // 🔵 FIELD YANG DIPANTAU
        $fields = [
            'nama_unit' => 'Nama Unit',
            'id_dosen'  => 'Dosen',
        ];

        foreach ($fields as $field => $label) {
            if (!isset($post[$field]) || !array_key_exists($field, $lama)) {
                continue;
            }

            if ($lama[$field] != $post[$field]) {

                // KHUSUS DOSEN → GANTI ID JADI NAMA
                if ($field === 'id_dosen') {
                    $lamaVal = $dosenLama['nama_lengkap'] ?? '-';
                    $baruVal = $dosenBaru['nama_lengkap'] ?? '-';
                } else {
                    $lamaVal = $lama[$field];
                    $baruVal = $post[$field];
                }

                $perubahan[] =
                    "<strong>{$label}</strong>:<br>" .
                    "Lama: \"" . esc($lamaVal) . "\"<br>" .
                    "Baru: \"" . esc($baruVal) . "\"";
            }
        }

        // 📝 KETERANGAN LOG
        $keterangan = empty($perubahan)
            ? 'Menyimpan data tanpa perubahan nilai'
            : implode('<hr>', $perubahan);

        // 🔵 UPDATE DATABASE
        $dataUpdate = [
            'nama_unit'  => $post['nama_unit'],
            'id_dosen'   => $post['id_dosen'],
            'updated_by' => session()->get('id'),
        ];

        $this->Unit_kerjaModel->update($id, $dataUpdate);

        // 🧾 LOG
        $user = $this->getUser();
        $this->logAktivitas->log(
            'update_unit_kerja',
            $user['id'],
            $user['name'],
            current_url(),
            $keterangan
        );

        return redirect()->to('/unit_kerja')->with('success', 'Unit kerja berhasil diperbarui.');
    }



    public function delete($id)
    {
        $permission = $this->checkPermission('unit_kerja_delete');
        if ($permission instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permission;
        }

        // 🔴 DATA LAMA
        $item = $this->Unit_kerjaModel->find($id);
        if (!$item) {
            return redirect()->to('/unit_kerja')->with('error', 'Data tidak ditemukan');
        }

        // 🔵 DOSEN
        $dosen = $this->DosenModel->find($item['id_dosen']);

        // 🗑️ DELETE
        $this->Unit_kerjaModel->delete($id);

        // 📝 LOG
        $user = $this->getUser();

        $keterangan = implode('<br>', [
            '<strong>Data yang dihapus:</strong>',
            '<strong>ID</strong>: ' . esc($id),
            '<strong>Nama Unit</strong>: ' . esc($item['nama_unit']),
            '<strong>Dosen</strong>: ' . esc($dosen['nama_lengkap'] ?? '-'),
        ]);

        $this->logAktivitas->log(
            'delete_unit_kerja',
            $user['id'],
            $user['name'],
            current_url(),
            $keterangan
        );

        return redirect()->to('/unit_kerja')->with('success', 'Unit kerja berhasil dihapus.');
    }




    public function show($id)
    {
        $permission = $this->checkPermission('unit_kerja_view');
        if ($permission instanceof \CodeIgniter\HTTP\RedirectResponse) return $permission;

        $data['item'] = $this->Unit_kerjaModel->find($id);

        return view('unit_kerja/show', $this->data + $data);
    }

    private function get_User()
    {
        return [
            'id'   => session()->get('id'),
            'name' => session()->get('name')
        ];
    }
    // --- FITUR EXCEL ---

    public function export()
    {
        $permissionCheck = $this->checkPermission('unit_kerja_view'); // Asumsi izin export sama dengan view
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data = $this->Unit_kerjaModel->findAll();
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Header Kolom
        $headers = ['id', 'id_unit_kerja', 'nama_unit', 'id_user', 'created_by', 'updated_by'];
        $capitalizedHeaders = array_map(fn($h) => ucwords(str_replace('_', ' ', $h)), $headers);
        $sheet->fromArray($capitalizedHeaders, NULL, 'A1');

        // Data
        $rowNumber = 2;
        foreach ($data as $item) {
            $rowData = [];
            foreach ($headers as $header) {
                $rowData[] = $item[$header] ?? '';
            }
            $sheet->fromArray($rowData, NULL, 'A' . $rowNumber++);
        }

        $writer = new Xlsx($spreadsheet);
        $filename = 'unit_kerja-export-' . date('Y-m-d') . '.xlsx';

        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header("Content-Disposition: attachment; filename=\"$filename\"");
        header('Cache-Control: max-age=0');
        $writer->save('php://output');
        exit;
    }

    public function import()
    {
        $permissionCheck = $this->checkPermission('unit_kerja_create'); // Asumsi izin import sama dengan create
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $file = $this->request->getFile('excel_file');
        if ($file && $file->isValid() && !$file->hasMoved()) {
            try {
                $spreadsheet = IOFactory::load($file->getTempName());
                $sheet = $spreadsheet->getActiveSheet();
                $dataRows = $sheet->toArray(null, true, true, true);

                if (count($dataRows) < 2) {
                    session()->setFlashdata('error', 'File Excel kosong atau hanya berisi header.');
                    return redirect()->to(base_url('/unit_kerja'));
                }

                $headerRow = array_values(array_shift($dataRows));
                $dbFields = array_map(fn($h) => strtolower(str_replace(' ', '_', $h)), $headerRow);

                $importData = [];
                foreach ($dataRows as $row) {
                    if (empty(array_filter(array_values($row)))) continue; // Skip baris kosong
                    $rowData = array_combine($dbFields, array_values($row));
                    $importData[] = $rowData;
                }

                if (!empty($importData)) {
                    $this->Unit_kerjaModel->insertBatch($importData);
                    session()->setFlashdata('success', count($importData) . ' baris data berhasil diimpor.');
                } else {
                    session()->setFlashdata('error', 'Tidak ada data valid untuk diimpor.');
                }
            } catch (\Exception $e) {
                session()->setFlashdata('error', 'Gagal memproses file Excel: ' . $e->getMessage());
            }
        } else {
            session()->setFlashdata('error', 'Gagal mengunggah file. Pastikan file valid.');
        }

        return redirect()->to(base_url('/unit_kerja'));
    }

    public function template()
    {
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Header kolom (hanya allowedFields)
        $headers = ['id_unit_kerja', 'nama_unit', 'id_user', 'created_by', 'updated_by'];
        $capitalizedHeaders = array_map(fn($h) => ucwords(str_replace('_', ' ', $h)), $headers);
        $sheet->fromArray($capitalizedHeaders, NULL, 'A1');

        $writer = new Xlsx($spreadsheet);
        $filename = 'unit_kerja-template.xlsx';

        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header("Content-Disposition: attachment; filename=\"$filename\"");
        header('Cache-Control: max-age=0');
        $writer->save('php://output');
        exit;
    }
}
