<?php

namespace App\Controllers;

use App\Libraries\LogAktivitas;
use App\Models\Temuan_auditModel;
use App\Models\Tindakan_koreksiModel;
use CodeIgniter\Controller;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\IOFactory;

class Tindakan_koreksiController extends BaseController
{
    protected $model;
    protected $temuanModel;
    protected $logAktivitas;

    public function __construct()
    {
        $this->model        = new Tindakan_koreksiModel();
        $this->temuanModel  = new Temuan_auditModel();
        $this->logAktivitas = new LogAktivitas();
    }

    /* =====================
     * INDEX
     * ===================== */
    public function index()
    {
        $permissionCheck = $this->checkPermission('tindakan_koreksi_view');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data['items'] = $this->model
            ->select('tindakan_koreksi.*, temuan_audit.hasil_temuan')
            ->join('temuan_audit', 'temuan_audit.id_temuan = tindakan_koreksi.id_temuan', 'left')
            ->findAll();

        return view('tindakan_koreksi/index', $this->data + $data);
    }

    /* =====================
     * CREATE
     * ===================== */
    public function create()
    {
        $permissionCheck = $this->checkPermission('tindakan_koreksi_create');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data['temuan'] = $this->temuanModel->findAll();
        return view('tindakan_koreksi/create', $this->data + $data);
    }

    public function store()
    {
        $permissionCheck = $this->checkPermission('tindakan_koreksi_create');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $post = $this->request->getPost();

        $data = [
            'id_temuan'        => $post['id_temuan'],
            'uraian_tindakan'  => $post['uraian_tindakan'],
            'penanggung_jawab' => $post['penanggung_jawab'],
            'tgl_rencana'      => $post['tgl_rencana'] ?? null,
            'tgl_realisasi'    => $post['tgl_realisasi'] ?? null,
            'status'           => empty($post['tgl_realisasi']) ? 'diajukan' : 'selesai',
            'created_by'       => session()->get('id'),
        ];

        $this->model->insert($data);
        $id = $this->model->getInsertID();

        $keterangan = implode('<br>', [
            '<strong>Data yang ditambahkan:</strong>',
            '<strong>ID Koreksi</strong>: ' . $id,
            '<strong>ID Temuan</strong>: ' . $data['id_temuan'],
            '<strong>Uraian</strong>: ' . nl2br(esc($data['uraian_tindakan'])),
            '<strong>Penanggung Jawab</strong>: ' . esc($data['penanggung_jawab']),
            '<strong>Status</strong>: ' . esc($data['status']),
        ]);

        $user = $this->getUser();
        $this->logAktivitas->log(
            'create_tindakan_koreksi',
            $user['id'],
            $user['name'],
            current_url(),
            $keterangan
        );

        return redirect()->to('/tindakan_koreksi')->with('success', 'Data berhasil ditambahkan.');
    }

    public function edit($id)
    {
        $permissionCheck = $this->checkPermission('tindakan_koreksi_edit');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $item = $this->model->find($id);
        if (!$item) {
            return redirect()->to('/tindakan_koreksi')->with('error', 'Data tidak ditemukan');
        }

        $data = [
            'item'   => $item,
            'temuan' => $this->temuanModel->findAll()
        ];

        return view('tindakan_koreksi/edit', $this->data + $data);
    }

    public function update($id)
    {
        $permissionCheck = $this->checkPermission('tindakan_koreksi_edit');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $lama = $this->model->find($id);
        if (!$lama) {
            return redirect()->to('/tindakan_koreksi')->with('error', 'Data tidak ditemukan');
        }

        $post = $this->request->getPost();

        $data = [
            'id_temuan'        => $post['id_temuan'],
            'uraian_tindakan'  => $post['uraian_tindakan'],
            'penanggung_jawab' => $post['penanggung_jawab'],
            'tgl_rencana'      => $post['tgl_rencana'] ?? null,
            'tgl_realisasi'    => $post['tgl_realisasi'] ?? null,
            'status'           => empty($post['tgl_realisasi']) ? $post['status'] : 'selesai',
            'updated_by'       => session()->get('id'),
        ];

        $perubahan = [];
        $labels = [
            'id_temuan'        => 'ID Temuan',
            'uraian_tindakan'  => 'Uraian Tindakan',
            'penanggung_jawab' => 'Penanggung Jawab',
            'tgl_rencana'      => 'Tanggal Rencana',
            'tgl_realisasi'    => 'Tanggal Realisasi',
            'status'           => 'Status',
        ];

        foreach ($labels as $field => $label) {
            if (isset($lama[$field]) && $lama[$field] != $data[$field]) {
                $perubahan[] =
                    "<strong>{$label}</strong>:<br>" .
                    "Lama: \"{$lama[$field]}\"<br>" .
                    "Baru: \"{$data[$field]}\"";
            }
        }

        $this->model->update($id, $data);

        $user = $this->getUser();
        $this->logAktivitas->log(
            'update_tindakan_koreksi',
            $user['id'],
            $user['name'],
            current_url(),
            empty($perubahan)
                ? 'Menyimpan data tanpa perubahan nilai'
                : implode('<hr>', $perubahan)
        );

        return redirect()->to('/tindakan_koreksi')->with('success', 'Data berhasil diperbarui.');
    }

    public function delete($id)
    {
        $permissionCheck = $this->checkPermission('tindakan_koreksi_delete');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $item = $this->model->find($id);
        if (!$item) {
            return redirect()->to('/tindakan_koreksi')->with('error', 'Data tidak ditemukan');
        }

        $this->model->delete($id);

        $keterangan = implode('<br>', [
            '<strong>Data yang dihapus:</strong>',
            '<strong>ID Koreksi</strong>: ' . $id,
            '<strong>ID Temuan</strong>: ' . $item['id_temuan'],
            '<strong>Uraian</strong>: ' . nl2br(esc($item['uraian_tindakan'])),
            '<strong>Penanggung Jawab</strong>: ' . esc($item['penanggung_jawab']),
            '<strong>Status</strong>: ' . esc($item['status']),
        ]);

        $user = $this->getUser();
        $this->logAktivitas->log(
            'delete_tindakan_koreksi',
            $user['id'],
            $user['name'],
            current_url(),
            $keterangan
        );

        return redirect()->to('/tindakan_koreksi')->with('success', 'Data berhasil dihapus.');
    }


    /* =====================
     * SHOW
     * ===================== */
    public function show($id)
    {
        $permissionCheck = $this->checkPermission('tindakan_koreksi_view');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $item = $this->model
            ->select('tindakan_koreksi.*, temuan_audit.hasil_temuan')
            ->join('temuan_audit', 'temuan_audit.id_temuan = tindakan_koreksi.id_temuan', 'left')
            ->where('id_koreksi', $id)
            ->first();

        if (!$item) {
            return redirect()->to('/tindakan_koreksi')->with('error', 'Data tidak ditemukan');
        }

        return view('tindakan_koreksi/show', $this->data + ['item' => $item]);
    }

    /* =====================
     * HELPER
     * ===================== */
    private function getUser()
    {
        return [
            'id'   => session()->get('id'),
            'name' => session()->get('name')
        ];
    }

    // --- FITUR EXCEL ---

    public function export()
    {
        $permissionCheck = $this->checkPermission('tindakan_koreksi_view'); // Asumsi izin export sama dengan view
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data = $this->model->findAll();
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Header Kolom
        $headers = ['id', 'id_koreksi', 'id_temuan', 'uraian_tindakan', 'penanggung_jawab', 'tgl_rencana', 'tgl_realisasi', 'status', 'created_by', 'updated_by'];
        $capitalizedHeaders = array_map(fn($h) => ucwords(str_replace('_', ' ', $h)), $headers);
        $sheet->fromArray($capitalizedHeaders, NULL, 'A1');

        // Data
        $rowNumber = 2;
        foreach ($data as $item) {
            $rowData = [];
            foreach ($headers as $header) {
                $rowData[] = $item[$header] ?? '';
            }
            $sheet->fromArray($rowData, NULL, 'A' . $rowNumber++);
        }

        $writer = new Xlsx($spreadsheet);
        $filename = 'tindakan_koreksi-export-' . date('Y-m-d') . '.xlsx';

        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header("Content-Disposition: attachment; filename=\"$filename\"");
        header('Cache-Control: max-age=0');
        $writer->save('php://output');
        exit;
    }

    public function import()
    {
        $permissionCheck = $this->checkPermission('tindakan_koreksi_create'); // Asumsi izin import sama dengan create
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $file = $this->request->getFile('excel_file');
        if ($file && $file->isValid() && !$file->hasMoved()) {
            try {
                $spreadsheet = IOFactory::load($file->getTempName());
                $sheet = $spreadsheet->getActiveSheet();
                $dataRows = $sheet->toArray(null, true, true, true);

                if (count($dataRows) < 2) {
                    session()->setFlashdata('error', 'File Excel kosong atau hanya berisi header.');
                    return redirect()->to(base_url('/tindakan_koreksi'));
                }

                $headerRow = array_values(array_shift($dataRows));
                $dbFields = array_map(fn($h) => strtolower(str_replace(' ', '_', $h)), $headerRow);

                $importData = [];
                foreach ($dataRows as $row) {
                    if (empty(array_filter(array_values($row)))) continue; // Skip baris kosong
                    $rowData = array_combine($dbFields, array_values($row));
                    $importData[] = $rowData;
                }

                if (!empty($importData)) {
                    $this->model->insertBatch($importData);
                    session()->setFlashdata('success', count($importData) . ' baris data berhasil diimpor.');
                } else {
                    session()->setFlashdata('error', 'Tidak ada data valid untuk diimpor.');
                }
            } catch (\Exception $e) {
                session()->setFlashdata('error', 'Gagal memproses file Excel: ' . $e->getMessage());
            }
        } else {
            session()->setFlashdata('error', 'Gagal mengunggah file. Pastikan file valid.');
        }

        return redirect()->to(base_url('/tindakan_koreksi'));
    }

    public function template()
    {
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Header kolom (hanya allowedFields)
        $headers = ['id_koreksi', 'id_temuan', 'uraian_tindakan', 'penanggung_jawab', 'tgl_rencana', 'tgl_realisasi', 'status', 'created_by', 'updated_by'];
        $capitalizedHeaders = array_map(fn($h) => ucwords(str_replace('_', ' ', $h)), $headers);
        $sheet->fromArray($capitalizedHeaders, NULL, 'A1');

        $writer = new Xlsx($spreadsheet);
        $filename = 'tindakan_koreksi-template.xlsx';

        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header("Content-Disposition: attachment; filename=\"$filename\"");
        header('Cache-Control: max-age=0');
        $writer->save('php://output');
        exit;
    }
}
