<?php

namespace App\Controllers;

use App\Models\Temuan_auditModel;
use App\Models\Penugasan_auditModel;
use CodeIgniter\Controller;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\IOFactory;
use App\Libraries\LogAktivitas;

class Temuan_auditController extends BaseController
{
    protected $temuanModel;
    protected $penugasanModel;
    protected $logAktivitas;


    public function __construct()
    {
        $this->temuanModel    = new Temuan_auditModel();
        $this->penugasanModel = new Penugasan_auditModel();
        $this->logAktivitas = new LogAktivitas();
    }

    // ================= INDEX =================
    public function index()
    {
        $session = session();
        if (!$session->get('isLoggedIn')) {
            return redirect()->to('/login');
        }

        $permissionCheck = $this->checkPermission('temuan_audit_view');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data['items'] = $this->temuanModel
            ->select('temuan_audit.*, penugasan_audit.id_penugasan')
            ->join('penugasan_audit', 'penugasan_audit.id_penugasan = temuan_audit.id_penugasan')
            ->findAll();

        return view('temuan_audit/index', $this->data + $data);
    }

    // ================= CREATE =================
    public function create()
    {
        $permissionCheck = $this->checkPermission('temuan_audit_create');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data['penugasan'] = $this->penugasanModel->findAll();
        return view('temuan_audit/create', $this->data + $data);
    }

    // ================= STORE =================
    public function store()
    {
        $permissionCheck = $this->checkPermission('temuan_audit_create');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $userId = session()->get('id');

        $data = [
            'id_penugasan' => $this->request->getPost('id_penugasan'),
            'hasil_temuan' => $this->request->getPost('hasil_temuan'),
            'temuan_audit' => $this->request->getPost('temuan_audit'),
            'created_by'   => $userId,
            'updated_by'   => $userId,
        ];

        $this->temuanModel->insert($data);
        $idTemuan = $this->temuanModel->getInsertID();

        $penugasan = $this->penugasanModel->find($data['id_penugasan']);

        $keterangan = implode('<br>', [
            '<strong>Data yang ditambahkan:</strong>',
            '<strong>ID Temuan</strong>: ' . $idTemuan,
            '<strong>ID Penugasan</strong>: ' . esc($penugasan['id_penugasan'] ?? '-'),
            '<strong>Hasil Temuan</strong>:<br>' . nl2br(esc($data['hasil_temuan'])),
            '<strong>Temuan Audit</strong>:<br>' . nl2br(esc($data['temuan_audit'])),
        ]);

        $user = $this->getUser();
        $this->logAktivitas->log(
            'create_temuan_audit',
            $user['id'],
            $user['name'],
            current_url(),
            $keterangan
        );

        return redirect()->to('/temuan_audit')
            ->with('success', 'Temuan audit berhasil ditambahkan');
    }


    // ================= EDIT =================
    public function edit($id)
    {
        $permissionCheck = $this->checkPermission('temuan_audit_edit');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data['item']      = $this->temuanModel->find($id);
        $data['penugasan'] = $this->penugasanModel->findAll();

        return view('temuan_audit/edit', $this->data + $data);
    }

    // ================= UPDATE =================
    public function update($id)
    {
        $permissionCheck = $this->checkPermission('temuan_audit_edit');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $lama = $this->temuanModel->find($id);
        if (!$lama) {
            return redirect()->to('/temuan_audit')
                ->with('error', 'Data tidak ditemukan');
        }

        $post = $this->request->getPost();
        $perubahan = [];

        $fields = [
            'id_penugasan' => 'Penugasan',
            'hasil_temuan' => 'Hasil Temuan',
            'temuan_audit' => 'Temuan Audit',
        ];

        foreach ($fields as $field => $label) {
            if (($lama[$field] ?? '') != ($post[$field] ?? '')) {

                $lamaValue = $lama[$field];
                $baruValue = $post[$field];

                if ($field === 'id_penugasan') {
                    $lamaValue = $this->penugasanModel->find($lama[$field])['id_penugasan'] ?? '-';
                    $baruValue = $this->penugasanModel->find($post[$field])['id_penugasan'] ?? '-';
                }

                $perubahan[] =
                    "<strong>{$label}</strong>:<br>" .
                    "Lama:<br>" . nl2br(esc($lamaValue)) . "<br>" .
                    "Baru:<br>" . nl2br(esc($baruValue));
            }
        }

        $data = [
            'id_penugasan' => $post['id_penugasan'],
            'hasil_temuan' => $post['hasil_temuan'],
            'temuan_audit' => $post['temuan_audit'],
            'updated_by'   => session()->get('id'),
        ];

        $this->temuanModel->update($id, $data);

        $keterangan = empty($perubahan)
            ? 'Menyimpan data tanpa perubahan nilai'
            : implode('<hr>', $perubahan);

        $user = $this->getUser();
        $this->logAktivitas->log(
            'update_temuan_audit',
            $user['id'],
            $user['name'],
            current_url(),
            $keterangan
        );

        return redirect()->to('/temuan_audit')
            ->with('success', 'Temuan audit berhasil diperbarui');
    }

    // ================= DELETE =================
    public function delete($id)
    {
        $permissionCheck = $this->checkPermission('temuan_audit_delete');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $item = $this->temuanModel->find($id);
        if (!$item) {
            return redirect()->to('/temuan_audit')
                ->with('error', 'Data tidak ditemukan');
        }

        $penugasan = $this->penugasanModel->find($item['id_penugasan']);

        $this->temuanModel->delete($id);

        $keterangan = implode('<br>', [
            '<strong>Data yang dihapus:</strong>',
            '<strong>ID Temuan</strong>: ' . $id,
            '<strong>ID Penugasan</strong>: ' . esc($penugasan['id_penugasan'] ?? '-'),
            '<strong>Hasil Temuan</strong>:<br>' . nl2br(esc($item['hasil_temuan'])),
            '<strong>Temuan Audit</strong>:<br>' . nl2br(esc($item['temuan_audit'])),
        ]);

        $user = $this->getUser();
        $this->logAktivitas->log(
            'delete_temuan_audit',
            $user['id'],
            $user['name'],
            current_url(),
            $keterangan
        );

        return redirect()->to('/temuan_audit')
            ->with('success', 'Data berhasil dihapus');
    }


    public function show($id)
    {
        $permissionCheck = $this->checkPermission('temuan_audit_view');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data['item'] = $this->temuanModel->find($id);
        return view('temuan_audit/show', $this->data + $data);
    }

    private function getUser()
    {
        return [
            'id'   => session()->get('id'),
            'name' => session()->get('name'),
        ];
    }
    // --- FITUR EXCEL ---

    public function export()
    {
        $permissionCheck = $this->checkPermission('temuan_audit_view'); // Asumsi izin export sama dengan view
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data = $this->temuanModel->findAll();
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Header Kolom
        $headers = ['id', 'id_temuan', 'hasil_temuan', 'id_penugasan', 'temuan_audit', 'created_by', 'updated_by'];
        $capitalizedHeaders = array_map(fn($h) => ucwords(str_replace('_', ' ', $h)), $headers);
        $sheet->fromArray($capitalizedHeaders, NULL, 'A1');

        // Data
        $rowNumber = 2;
        foreach ($data as $item) {
            $rowData = [];
            foreach ($headers as $header) {
                $rowData[] = $item[$header] ?? '';
            }
            $sheet->fromArray($rowData, NULL, 'A' . $rowNumber++);
        }

        $writer = new Xlsx($spreadsheet);
        $filename = 'temuan_audit-export-' . date('Y-m-d') . '.xlsx';

        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header("Content-Disposition: attachment; filename=\"$filename\"");
        header('Cache-Control: max-age=0');
        $writer->save('php://output');
        exit;
    }

    public function import()
    {
        $permissionCheck = $this->checkPermission('temuan_audit_create'); // Asumsi izin import sama dengan create
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $file = $this->request->getFile('excel_file');
        if ($file && $file->isValid() && !$file->hasMoved()) {
            try {
                $spreadsheet = IOFactory::load($file->getTempName());
                $sheet = $spreadsheet->getActiveSheet();
                $dataRows = $sheet->toArray(null, true, true, true);

                if (count($dataRows) < 2) {
                    session()->setFlashdata('error', 'File Excel kosong atau hanya berisi header.');
                    return redirect()->to(base_url('/temuan_audit'));
                }

                $headerRow = array_values(array_shift($dataRows));
                $dbFields = array_map(fn($h) => strtolower(str_replace(' ', '_', $h)), $headerRow);

                $importData = [];
                foreach ($dataRows as $row) {
                    if (empty(array_filter(array_values($row)))) continue; // Skip baris kosong
                    $rowData = array_combine($dbFields, array_values($row));
                    $importData[] = $rowData;
                }

                if (!empty($importData)) {
                    $this->temuanModel->insertBatch($importData);
                    session()->setFlashdata('success', count($importData) . ' baris data berhasil diimpor.');
                } else {
                    session()->setFlashdata('error', 'Tidak ada data valid untuk diimpor.');
                }
            } catch (\Exception $e) {
                session()->setFlashdata('error', 'Gagal memproses file Excel: ' . $e->getMessage());
            }
        } else {
            session()->setFlashdata('error', 'Gagal mengunggah file. Pastikan file valid.');
        }

        return redirect()->to(base_url('/temuan_audit'));
    }

    public function template()
    {
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Header kolom (hanya allowedFields)
        $headers = ['id_temuan', 'hasil_temuan', 'id_penugasan', 'temuan_audit', 'created_by', 'updated_by'];
        $capitalizedHeaders = array_map(fn($h) => ucwords(str_replace('_', ' ', $h)), $headers);
        $sheet->fromArray($capitalizedHeaders, NULL, 'A1');

        $writer = new Xlsx($spreadsheet);
        $filename = 'temuan_audit-template.xlsx';

        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header("Content-Disposition: attachment; filename=\"$filename\"");
        header('Cache-Control: max-age=0');
        $writer->save('php://output');
        exit;
    }
}
