<?php

namespace App\Controllers;

use App\Models\Standar_lembaga_akreditasiModel;
use App\Models\Lemb_akreditasiModel;
use CodeIgniter\Controller;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\IOFactory;
use App\Libraries\LogAktivitas;


class Standar_lembaga_akreditasiController extends BaseController
{
    protected $StandarModel;
    protected $LembagaModel;
    protected $logAktivitas;

    public function __construct()
    {
        $this->StandarModel = new Standar_lembaga_akreditasiModel();
        $this->LembagaModel = new Lemb_akreditasiModel();
        $this->logAktivitas = new LogAktivitas();
    }

    /* ======================
       INDEX
    ====================== */
    public function index()
    {
        if (!session()->get('isLoggedIn')) {
            return redirect()->to('/login');
        }

        $permissionCheck = $this->checkPermission('standar_lembaga_akreditasi_view');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data['items'] = $this->StandarModel->findAll();

        return view('standar_lembaga_akreditasi/index', $this->data + $data);
    }


    public function create()
    {
        $permissionCheck = $this->checkPermission('standar_lembaga_akreditasi_create');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data['lembaga'] = $this->LembagaModel->findAll();

        return view('standar_lembaga_akreditasi/create', $this->data + $data);
    }


    public function store()
    {
        $permissionCheck = $this->checkPermission('standar_lembaga_akreditasi_create');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $last = $this->StandarModel->orderBy('id_standar', 'DESC')->first();
        $nextNumber = $last ? intval(substr($last['id_standar'], 4)) + 1 : 1;
        $idStandar = 'STD-' . str_pad($nextNumber, 3, '0', STR_PAD_LEFT);

        $data = [
            'id_standar'   => $idStandar,
            'id_lemb'      => $this->request->getPost('id_lemb'),
            'kode_standar' => $this->request->getPost('kode_standar'),
            'nama_standar' => $this->request->getPost('nama_standar'),
            'deskripsi'    => $this->request->getPost('deskripsi'),
            'created_by'   => session()->get('id'),
            'updated_by'   => session()->get('id'),
        ];

        $this->StandarModel->insert($data);

        $lembaga = $this->LembagaModel->find($data['id_lemb']);

        $keterangan = implode('<br>', [
            '<strong>Data yang ditambahkan:</strong>',
            '<strong>ID Standar</strong>: ' . esc($idStandar),
            '<strong>Lembaga</strong>: ' . esc($lembaga['nama_lembaga'] ?? '-'),
            '<strong>Kode Standar</strong>: ' . esc($data['kode_standar']),
            '<strong>Nama Standar</strong>: ' . esc($data['nama_standar']),
            '<strong>Deskripsi</strong>:<br>' . nl2br(esc($data['deskripsi'])),
        ]);

        $user = $this->getUser();
        $this->logAktivitas->log(
            'create_standar_lembaga',
            $user['id'],
            $user['name'],
            current_url(),
            $keterangan
        );

        return redirect()->to('/standar_lembaga_akreditasi')
            ->with('success', 'Standar lembaga akreditasi berhasil ditambahkan.');
    }


    public function edit($id)
    {
        $permissionCheck = $this->checkPermission('standar_lembaga_akreditasi_edit');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data['item']    = $this->StandarModel->find($id);
        $data['lembaga'] = $this->LembagaModel->findAll();

        return view('standar_lembaga_akreditasi/edit', $this->data + $data);
    }

    public function update($id)
    {
        $permissionCheck = $this->checkPermission('standar_lembaga_akreditasi_edit');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $lama = $this->StandarModel->find($id);
        if (!$lama) {
            return redirect()->to('/standar_lembaga_akreditasi')
                ->with('error', 'Data tidak ditemukan');
        }

        $post = $this->request->getPost();

        $perubahan = [];

        $fields = [
            'id_lemb'      => 'Lembaga',
            'kode_standar' => 'Kode Standar',
            'nama_standar' => 'Nama Standar',
            'deskripsi'    => 'Deskripsi',
        ];

        foreach ($fields as $field => $label) {
            if (($lama[$field] ?? '') != ($post[$field] ?? '')) {

                $lamaValue = $lama[$field];
                $baruValue = $post[$field];

                if ($field === 'id_lemb') {
                    $lamaValue = $this->LembagaModel->find($lama[$field])['nama_lembaga'] ?? '-';
                    $baruValue = $this->LembagaModel->find($post[$field])['nama_lembaga'] ?? '-';
                }

                $perubahan[] =
                    "<strong>{$label}</strong>:<br>" .
                    "Lama:<br>" . nl2br(esc($lamaValue)) . "<br>" .
                    "Baru:<br>" . nl2br(esc($baruValue));
            }
        }

        $data = [
            'id_lemb'      => $post['id_lemb'],
            'kode_standar' => $post['kode_standar'],
            'nama_standar' => $post['nama_standar'],
            'deskripsi'    => $post['deskripsi'],
            'updated_by'   => session()->get('id'),
        ];

        $this->StandarModel->update($id, $data);

        $keterangan = empty($perubahan)
            ? 'Menyimpan data tanpa perubahan nilai'
            : implode('<hr>', $perubahan);

        $user = $this->getUser();
        $this->logAktivitas->log(
            'update_standar_lembaga',
            $user['id'],
            $user['name'],
            current_url(),
            $keterangan
        );

        return redirect()->to('/standar_lembaga_akreditasi')
            ->with('success', 'Standar lembaga akreditasi berhasil diperbarui.');
    }



    public function delete($id)
    {
        $permissionCheck = $this->checkPermission('standar_lembaga_akreditasi_delete');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $item = $this->StandarModel->find($id);
        if (!$item) {
            return redirect()->to('/standar_lembaga_akreditasi')
                ->with('error', 'Data tidak ditemukan');
        }

        $lembaga = $this->LembagaModel->find($item['id_lemb']);

        $this->StandarModel->delete($id);

        $keterangan = implode('<br>', [
            '<strong>Data yang dihapus:</strong>',
            '<strong>ID Standar</strong>: ' . esc($item['id_standar']),
            '<strong>Lembaga</strong>: ' . esc($lembaga['nama_lembaga'] ?? '-'),
            '<strong>Kode Standar</strong>: ' . esc($item['kode_standar']),
            '<strong>Nama Standar</strong>: ' . esc($item['nama_standar']),
            '<strong>Deskripsi</strong>:<br>' . nl2br(esc($item['deskripsi'])),
        ]);

        $user = $this->getUser();
        $this->logAktivitas->log(
            'delete_standar_lembaga',
            $user['id'],
            $user['name'],
            current_url(),
            $keterangan
        );

        return redirect()->to('/standar_lembaga_akreditasi')
            ->with('success', 'Data berhasil dihapus.');
    }



    public function show($id)
    {
        $permissionCheck = $this->checkPermission('standar_lembaga_akreditasi_view');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $item = $this->StandarModel->find($id);
        $data['item'] = $this->StandarModel->find($id);
        return view('standar_lembaga_akreditasi/show', $this->data + $data);
    }

    private function getUser()
    {
        return [
            'id'   => session()->get('id'),
            'name' => session()->get('name'),
        ];
    }

    // --- FITUR EXCEL ---

    public function export()
    {
        $permissionCheck = $this->checkPermission('standar_lembaga_akreditasi_view'); // Asumsi izin export sama dengan view
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data = $this->StandarModel->findAll();
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Header Kolom
        $headers = ['id', 'id_standar', 'id_lemb', 'kode_standar', 'nama_standar', 'deskripsi', 'created_by', 'updated_by'];
        $capitalizedHeaders = array_map(fn($h) => ucwords(str_replace('_', ' ', $h)), $headers);
        $sheet->fromArray($capitalizedHeaders, NULL, 'A1');

        // Data
        $rowNumber = 2;
        foreach ($data as $item) {
            $rowData = [];
            foreach ($headers as $header) {
                $rowData[] = $item[$header] ?? '';
            }
            $sheet->fromArray($rowData, NULL, 'A' . $rowNumber++);
        }

        $writer = new Xlsx($spreadsheet);
        $filename = 'standar_lembaga_akreditasi-export-' . date('Y-m-d') . '.xlsx';

        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header("Content-Disposition: attachment; filename=\"$filename\"");
        header('Cache-Control: max-age=0');
        $writer->save('php://output');
        exit;
    }

    public function import()
    {
        $permissionCheck = $this->checkPermission('standar_lembaga_akreditasi_create'); // Asumsi izin import sama dengan create
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $file = $this->request->getFile('excel_file');
        if ($file && $file->isValid() && !$file->hasMoved()) {
            try {
                $spreadsheet = IOFactory::load($file->getTempName());
                $sheet = $spreadsheet->getActiveSheet();
                $dataRows = $sheet->toArray(null, true, true, true);

                if (count($dataRows) < 2) {
                    session()->setFlashdata('error', 'File Excel kosong atau hanya berisi header.');
                    return redirect()->to(base_url('/standar_lembaga_akreditasi'));
                }

                $headerRow = array_values(array_shift($dataRows));
                $dbFields = array_map(fn($h) => strtolower(str_replace(' ', '_', $h)), $headerRow);

                $importData = [];
                foreach ($dataRows as $row) {
                    if (empty(array_filter(array_values($row)))) continue; // Skip baris kosong
                    $rowData = array_combine($dbFields, array_values($row));
                    $importData[] = $rowData;
                }

                if (!empty($importData)) {
                    $this->StandarModel->insertBatch($importData);
                    session()->setFlashdata('success', count($importData) . ' baris data berhasil diimpor.');
                } else {
                    session()->setFlashdata('error', 'Tidak ada data valid untuk diimpor.');
                }
            } catch (\Exception $e) {
                session()->setFlashdata('error', 'Gagal memproses file Excel: ' . $e->getMessage());
            }
        } else {
            session()->setFlashdata('error', 'Gagal mengunggah file. Pastikan file valid.');
        }

        return redirect()->to(base_url('/standar_lembaga_akreditasi'));
    }

    public function template()
    {
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Header kolom (hanya allowedFields)
        $headers = ['id_standar', 'id_lemb', 'kode_standar', 'nama_standar', 'deskripsi', 'created_by', 'updated_by'];
        $capitalizedHeaders = array_map(fn($h) => ucwords(str_replace('_', ' ', $h)), $headers);
        $sheet->fromArray($capitalizedHeaders, NULL, 'A1');

        $writer = new Xlsx($spreadsheet);
        $filename = 'standar_lembaga_akreditasi-template.xlsx';

        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header("Content-Disposition: attachment; filename=\"$filename\"");
        header('Cache-Control: max-age=0');
        $writer->save('php://output');
        exit;
    }
}
