<?php

namespace App\Controllers;

use App\Models\RoleModel;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\IOFactory;
use App\Libraries\LogAktivitas;

class RoleController extends BaseController
{
    protected $model;
    protected $logAktivitas;

    public function __construct()
    {
        parent::initController(service('request'), service('response'), service('logger'));
        $this->model = new RoleModel();
        $this->logAktivitas = new LogAktivitas();
    }

    public function index()
    {
        if (!session()->get('isLoggedIn')) {
            return redirect()->to('/login');
        }

        $permissionCheck = $this->checkPermission('roles_view');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data = $this->data;
        $data['items'] = $this->model->findAll();

        $user = $this->getUser();
        $this->logAktivitas->log('akses_role', $user['id'], $user['name'], current_url());

        return view('roles/index', $data);
    }

    public function create()
    {
        $permissionCheck = $this->checkPermission('roles_create');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        return view('roles/create', $this->data);
    }

    public function store()
    {
        $permissionCheck = $this->checkPermission('roles_create');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        try {
            $data = $this->normalizeInput($this->request->getPost());
            if (isset($data['permissions']) && is_array($data['permissions'])) {
                $data['permissions'] = json_encode($data['permissions']);
            }

            $this->model->insert($data);

            $user = $this->getUser();
            $this->logAktivitas->log('create_role', $user['id'], $user['name'], current_url());

            return redirect()->to('/roles')->with('success', 'Data berhasil ditambahkan.');
        } catch (\Throwable $e) {
            return redirect()->back()->with('error', 'Terjadi kesalahan: ' . $e->getMessage());
        }
    }

    public function edit($id)
    {
        $permissionCheck = $this->checkPermission('roles_edit');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data = $this->data;
        $data['item'] = $this->model->find($id);

        $user = $this->getUser();
        $this->logAktivitas->log('edit_role', $user['id'], $user['name'], current_url());

        return view('roles/edit', $data);
    }

    public function update($id)
    {
        $permissionCheck = $this->checkPermission('roles_edit');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        try {
            $data = $this->normalizeInput($this->request->getPost());

            if (isset($data['permissions']) && is_array($data['permissions'])) {
                $data['permissions'] = json_encode($data['permissions']);
            }

            $this->model->update($id, $data);

            $user = $this->getUser();
            $this->logAktivitas->log('update_role', $user['id'], $user['name'], current_url());

            return redirect()->to('/roles')->with('success', 'Data berhasil diperbarui.');
        } catch (\Throwable $e) {
            return redirect()->back()->with('error', 'Terjadi kesalahan: ' . $e->getMessage());
        }
    }

    public function delete($id)
    {
        $permissionCheck = $this->checkPermission('roles_delete');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $this->model->delete($id);

        $user = $this->getUser();
        $this->logAktivitas->log('delete_role', $user['id'], $user['name'], current_url());

        return redirect()->to('/roles')->with('success', 'Data berhasil dihapus.');
    }

    public function show($id)
    {
        $permissionCheck = $this->checkPermission('roles_view');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data = $this->data;
        $data['item'] = $this->model->find($id);

        return view('roles/show', $data);
    }

    // =======================
    // HELPER
    // =======================
    private function getUser()
    {
        return [
            'id'   => session()->get('id'),
            'name' => session()->get('name'),
        ];
    }

    private function normalizeInput($data)
    {
        if (!is_array($data)) {
            $tmp = json_decode($data, true);
            $data = is_array($tmp) ? $tmp : [];
        }

        if (isset($data['permissions'])) {
            if (is_string($data['permissions'])) {
                $data['permissions'] = json_decode($data['permissions'], true) ?? [];
            }
        } else {
            $data['permissions'] = [];
        }

        return $data;
    }
    // ===============================
    // 📦 Fitur Excel
    // ===============================
    public function export()
    {
        $permissionCheck = $this->checkPermission('roles_view');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data = $this->model->findAll();
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        $headers = ['id', 'name', 'description', 'permissions'];
        $sheet->fromArray(array_map('ucwords', $headers), NULL, 'A1');

        $row = 2;
        foreach ($data as $item) {
            $sheet->fromArray([
                $item['id'],
                $item['name'],
                $item['description'],
                is_array($item['permissions']) ? implode(',', $item['permissions']) : $item['permissions']
            ], NULL, 'A' . $row++);
        }

        $writer = new Xlsx($spreadsheet);
        $filename = 'roles-export-' . date('Y-m-d') . '.xlsx';
        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header("Content-Disposition: attachment; filename=\"$filename\"");
        header('Cache-Control: max-age=0');
        $writer->save('php://output');
        exit;
    }

    public function import()
    {
        $permissionCheck = $this->checkPermission('roles_create');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $file = $this->request->getFile('excel_file');
        if ($file && $file->isValid() && !$file->hasMoved()) {
            try {
                $spreadsheet = IOFactory::load($file->getTempName());
                $sheet = $spreadsheet->getActiveSheet();
                $rows = $sheet->toArray(null, true, true, true);
                array_shift($rows);

                $insertData = [];
                foreach ($rows as $r) {
                    if (empty(array_filter($r))) continue;
                    $insertData[] = [
                        'name' => $r['A'] ?? '',
                        'description' => $r['B'] ?? '',
                        'permissions' => array_map('trim', explode(',', $r['C'] ?? '')),
                    ];
                }

                if (!empty($insertData)) {
                    $this->model->insertBatch($insertData);
                    return redirect()->to('/roles')->with('success', count($insertData) . ' data berhasil diimpor.');
                }
            } catch (\Exception $e) {
                return redirect()->to('/roles')->with('error', 'Gagal import: ' . $e->getMessage());
            }
        }

        return redirect()->to('/roles')->with('error', 'File tidak valid.');
    }

    public function template()
    {
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();
        $sheet->fromArray(['Name', 'Description', 'Permissions (comma separated)'], NULL, 'A1');

        $writer = new Xlsx($spreadsheet);
        $filename = 'roles-template.xlsx';
        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header("Content-Disposition: attachment; filename=\"$filename\"");
        header('Cache-Control: max-age=0');
        $writer->save('php://output');
        exit;
    }
}
