<?php

namespace App\Controllers;

use App\Models\ProdiModel;
use CodeIgniter\Controller;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\IOFactory;
use App\Libraries\LogAktivitas;


class ProdiController extends BaseController
{
    protected $ProdiModel;
    protected $logAktivitas;

    public function __construct()
    {
        $this->ProdiModel = new ProdiModel();
        $this->logAktivitas = new LogAktivitas();
    }

    public function index()
    {
        $session = session();
        if (!$session->get('isLoggedIn')) {
            return redirect()->to('/login');
        }

        $permissionCheck = $this->checkPermission('prodi_view');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data['items'] = $this->ProdiModel->findAll();
        $user = $this->getUser();
        $this->logAktivitas->log('akses_prodi', $user['id'], $user['name'], current_url());
        return view('prodi/index', $this->data + $data);
    }

    public function create()
    {
        $permissionCheck = $this->checkPermission('prodi_create');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        return view('prodi/create', $this->data);
    }

    public function store()
    {
        $permissionCheck = $this->checkPermission('prodi_create');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $user = $this->getUser();

        $data = [
            'nama_prodi' => $this->request->getPost('nama_prodi'),
            'created_by' => $user['id'],
            'updated_by' => $user['id'],
            'id_user'    => $user['id'],
        ];

        // INSERT
        $this->ProdiModel->insert($data);

        // ID ASLI DATABASE
        $idBaru = $this->ProdiModel->getInsertID();

        // 🔎 REKAP
        $keterangan = [];
        $keterangan[] = '<strong>Data yang ditambahkan:</strong>';
        $keterangan[] = '<strong>ID</strong>: ' . $idBaru;
        $keterangan[] = '<strong>Nama Prodi</strong>: ' . esc($data['nama_prodi']);

        // LOG
        $this->logAktivitas->log(
            'create_prodi',
            $user['id'],
            $user['name'],
            current_url(),
            implode('<br>', $keterangan)
        );

        return redirect()->to('/prodi')->with('success', 'Prodi berhasil ditambahkan.');
    }



    public function edit($id)
    {
        $permissionCheck = $this->checkPermission('prodi_edit');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data['item'] = $this->ProdiModel->find($id);
        $user = $this->getUser();
        $this->logAktivitas->log('edit_prodi', $user['id'], $user['name'], current_url());
        return view('prodi/edit', $this->data + $data);
    }

    public function update($id)
    {
        $permissionCheck = $this->checkPermission('prodi_edit');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $user = $this->getUser();

        // DATA LAMA
        $old = $this->ProdiModel->find($id);

        $data = [
            'nama_prodi' => $this->request->getPost('nama_prodi'),
            'updated_by' => $user['id'],
        ];

        $this->ProdiModel->update($id, $data);

        // 🔎 REKAP PERUBAHAN
        $keterangan = [];
        $keterangan[] = '<strong>Data yang diperbarui:</strong>';
        $keterangan[] = '<strong>ID</strong>: ' . $id;

        if ($old['nama_prodi'] !== $data['nama_prodi']) {
            $keterangan[] =
                '<strong>Nama Prodi</strong>:<br>' .
                'Sebelum: "' . esc($old['nama_prodi']) . '"<br>' .
                'Sesudah: "' . esc($data['nama_prodi']) . '"';
        }

        $this->logAktivitas->log(
            'update_prodi',
            $user['id'],
            $user['name'],
            current_url(),
            implode('<hr>', $keterangan)
        );

        return redirect()->to('/prodi')->with('success', 'Data berhasil diperbarui.');
    }


    public function delete($id)
    {
        $permissionCheck = $this->checkPermission('prodi_delete');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $user = $this->getUser();

        // DATA SEBELUM DIHAPUS
        $item = $this->ProdiModel->find($id);

        $this->ProdiModel->delete($id);

        // 🔎 REKAP
        $keterangan = [];
        $keterangan[] = '<strong>Data yang dihapus:</strong>';
        $keterangan[] = '<strong>ID</strong>: ' . $id;
        $keterangan[] = '<strong>Nama Prodi</strong>: ' . esc($item['nama_prodi']);

        $this->logAktivitas->log(
            'delete_prodi',
            $user['id'],
            $user['name'],
            current_url(),
            implode('<br>', $keterangan)
        );

        return redirect()->to('/prodi')->with('success', 'Data berhasil dihapus.');
    }


    public function show($id)
    {
        $permissionCheck = $this->checkPermission('prodi_view');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data['item'] = $this->ProdiModel->find($id);
        return view('prodi/show', $data);
    }

    private function getUser()
    {
        return [
            'id'   => session()->get('id'),
            'name' => session()->get('name'),
        ];
    }
    // --- FITUR EXCEL ---

    public function export()
    {
        $permissionCheck = $this->checkPermission('prodi_view'); // Asumsi izin export sama dengan view
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data = $this->ProdiModel->findAll();
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Header Kolom
        $headers = ['id', 'id_prodi', 'nama_prodi', 'created_by', 'updated_by', 'id_user'];
        $capitalizedHeaders = array_map(fn($h) => ucwords(str_replace('_', ' ', $h)), $headers);
        $sheet->fromArray($capitalizedHeaders, NULL, 'A1');

        // Data
        $rowNumber = 2;
        foreach ($data as $item) {
            $rowData = [];
            foreach ($headers as $header) {
                $rowData[] = $item[$header] ?? '';
            }
            $sheet->fromArray($rowData, NULL, 'A' . $rowNumber++);
        }

        $writer = new Xlsx($spreadsheet);
        $filename = 'prodi-export-' . date('Y-m-d') . '.xlsx';

        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header("Content-Disposition: attachment; filename=\"$filename\"");
        header('Cache-Control: max-age=0');
        $writer->save('php://output');
        exit;
    }

    public function import()
    {
        $permissionCheck = $this->checkPermission('prodi_create'); // Asumsi izin import sama dengan create
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $file = $this->request->getFile('excel_file');
        if ($file && $file->isValid() && !$file->hasMoved()) {
            try {
                $spreadsheet = IOFactory::load($file->getTempName());
                $sheet = $spreadsheet->getActiveSheet();
                $dataRows = $sheet->toArray(null, true, true, true);

                if (count($dataRows) < 2) {
                    session()->setFlashdata('error', 'File Excel kosong atau hanya berisi header.');
                    return redirect()->to(base_url('/prodi'));
                }

                $headerRow = array_values(array_shift($dataRows));
                $dbFields = array_map(fn($h) => strtolower(str_replace(' ', '_', $h)), $headerRow);

                $importData = [];
                foreach ($dataRows as $row) {
                    if (empty(array_filter(array_values($row)))) continue; // Skip baris kosong
                    $rowData = array_combine($dbFields, array_values($row));
                    $importData[] = $rowData;
                }

                if (!empty($importData)) {
                    $this->ProdiModel->insertBatch($importData);
                    session()->setFlashdata('success', count($importData) . ' baris data berhasil diimpor.');
                } else {
                    session()->setFlashdata('error', 'Tidak ada data valid untuk diimpor.');
                }
            } catch (\Exception $e) {
                session()->setFlashdata('error', 'Gagal memproses file Excel: ' . $e->getMessage());
            }
        } else {
            session()->setFlashdata('error', 'Gagal mengunggah file. Pastikan file valid.');
        }

        return redirect()->to(base_url('/prodi'));
    }

    public function template()
    {
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Header kolom (hanya allowedFields)
        $headers = ['id_prodi', 'nama_prodi', 'created_by', 'updated_by', 'id_user'];
        $capitalizedHeaders = array_map(fn($h) => ucwords(str_replace('_', ' ', $h)), $headers);
        $sheet->fromArray($capitalizedHeaders, NULL, 'A1');

        $writer = new Xlsx($spreadsheet);
        $filename = 'prodi-template.xlsx';

        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header("Content-Disposition: attachment; filename=\"$filename\"");
        header('Cache-Control: max-age=0');
        $writer->save('php://output');
        exit;
    }
}
