<?php

namespace App\Controllers;

use App\Models\Periode_auditModel;
use CodeIgniter\Controller;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\IOFactory;
use App\Libraries\LogAktivitas;

class Periode_auditController extends BaseController
{
    protected $Periode_auditModel;
    protected $logAktivitas;

    public function __construct()
    {
        $this->Periode_auditModel = new Periode_auditModel();
        $this->logAktivitas = new LogAktivitas();
    }

    private function getUser()
    {
        return [
            'id'   => session()->get('id'),
            'name' => session()->get('name'),
        ];
    }

    public function index()
    {
        $session = session();
        if (!$session->get('isLoggedIn')) {
            return redirect()->to('/login');
        }

        $permissionCheck = $this->checkPermission('periode_audit_view');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data['items'] = $this->Periode_auditModel->findAll();
        $user = $this->getUser();
        $this->logAktivitas->log('akses_periode_audit', $user['id'], $user['name'], current_url());

        return view('periode_audit/index', $this->data + $data);
    }

    public function create()
    {
        $permissionCheck = $this->checkPermission('periode_audit_create');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        return view('periode_audit/create', $this->data);
    }

    public function store()
    {
        if (!session()->get('isLoggedIn')) {
            return redirect()->to('/login');
        }

        $permissionCheck = $this->checkPermission('periode_audit_create');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $user = $this->getUser();

        $data = [
            'tahun_audit'    => $this->request->getPost('tahun_audit'),
            'tgl_mulai'      => $this->request->getPost('tgl_mulai'),
            'tgl_selesai'    => $this->request->getPost('tgl_selesai'),
            'status_periode' => $this->request->getPost('status_periode'),
            'created_by'     => $user['id'],
            'updated_by'     => $user['id'],
        ];

        // INSERT
        $this->Periode_auditModel->insert($data);
        $idBaru = $this->Periode_auditModel->getInsertID();

        // 🔎 REKAP
        $keterangan = [];
        $keterangan[] = '<strong>Data yang ditambahkan:</strong>';
        $keterangan[] = '<strong>ID</strong>: ' . $idBaru;
        $keterangan[] = '<strong>Tahun Audit</strong>: ' . esc($data['tahun_audit']);
        $keterangan[] = '<strong>Tanggal Mulai</strong>: ' . esc($data['tgl_mulai']);
        $keterangan[] = '<strong>Tanggal Selesai</strong>: ' . esc($data['tgl_selesai']);
        $keterangan[] = '<strong>Status</strong>: ' . esc($data['status_periode']);

        // LOG
        $this->logAktivitas->log(
            'create_periode_audit',
            $user['id'],
            $user['name'],
            current_url(),
            implode('<br>', $keterangan)
        );

        return redirect()->to('/periode_audit')->with('success', 'Data berhasil ditambahkan');
    }



    public function edit($id)
    {
        $permissionCheck = $this->checkPermission('periode_audit_edit');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data['item'] = $this->Periode_auditModel->find($id);
        $user = $this->getUser();
        $this->logAktivitas->log('edit_periode_audit', $user['id'], $user['name'], current_url());

        return view('periode_audit/edit', $this->data + $data);
    }

    public function update($id)
    {
        $permissionCheck = $this->checkPermission('periode_audit_edit');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $user = $this->getUser();

        // DATA LAMA
        $old = $this->Periode_auditModel->find($id);

        $data = [
            'tahun_audit'    => $this->request->getPost('tahun_audit'),
            'tgl_mulai'      => $this->request->getPost('tgl_mulai'),
            'tgl_selesai'    => $this->request->getPost('tgl_selesai'),
            'status_periode' => $this->request->getPost('status_periode'),
            'updated_by'     => $user['id'],
        ];

        $this->Periode_auditModel->update($id, $data);

        // 🔎 REKAP PERUBAHAN
        $keterangan = [];
        $keterangan[] = '<strong>Data yang diperbarui:</strong>';
        $keterangan[] = '<strong>ID</strong>: ' . $id;

        foreach ($data as $field => $value) {
            if (isset($old[$field]) && $old[$field] != $value) {
                $keterangan[] =
                    '<strong>' . ucwords(str_replace('_', ' ', $field)) . '</strong>:<br>' .
                    'Sebelum: "' . esc($old[$field]) . '"<br>' .
                    'Sesudah: "' . esc($value) . '"';
            }
        }

        $this->logAktivitas->log(
            'update_periode_audit',
            $user['id'],
            $user['name'],
            current_url(),
            implode('<hr>', $keterangan)
        );

        return redirect()->to('/periode_audit')->with('success', 'Data berhasil diperbarui');
    }


    public function delete($id)
    {
        $permissionCheck = $this->checkPermission('periode_audit_delete');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $user = $this->getUser();

        // DATA SEBELUM DIHAPUS
        $item = $this->Periode_auditModel->find($id);

        $this->Periode_auditModel->delete($id);

        // 🔎 REKAP
        $keterangan = [];
        $keterangan[] = '<strong>Data yang dihapus:</strong>';
        $keterangan[] = '<strong>ID</strong>: ' . $id;
        $keterangan[] = '<strong>Tahun Audit</strong>: ' . esc($item['tahun_audit']);
        $keterangan[] = '<strong>Status</strong>: ' . esc($item['status_periode']);

        $this->logAktivitas->log(
            'delete_periode_audit',
            $user['id'],
            $user['name'],
            current_url(),
            implode('<br>', $keterangan)
        );

        return redirect()->to('/periode_audit')->with('success', 'Data berhasil dihapus');
    }



    public function show($id)
    {
        $permissionCheck = $this->checkPermission('periode_audit_view');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data['item'] = $this->Periode_auditModel->find($id);
        return view('periode_audit/show', $this->data + $data);
    }

    // --- FITUR EXCEL ---

    public function export()
    {
        $permissionCheck = $this->checkPermission('periode_audit_view'); // Asumsi izin export sama dengan view
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data = $this->Periode_auditModel->findAll();
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Header Kolom
        $headers = ['id', 'id_periode', 'tahun_audit', 'tgl_mulai', 'tgl_selesai', 'status_periode', 'created_by', 'updated_by'];
        $capitalizedHeaders = array_map(fn($h) => ucwords(str_replace('_', ' ', $h)), $headers);
        $sheet->fromArray($capitalizedHeaders, NULL, 'A1');

        // Data
        $rowNumber = 2;
        foreach ($data as $item) {
            $rowData = [];
            foreach ($headers as $header) {
                $rowData[] = $item[$header] ?? '';
            }
            $sheet->fromArray($rowData, NULL, 'A' . $rowNumber++);
        }

        $writer = new Xlsx($spreadsheet);
        $filename = 'periode_audit-export-' . date('Y-m-d') . '.xlsx';

        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header("Content-Disposition: attachment; filename=\"$filename\"");
        header('Cache-Control: max-age=0');
        $writer->save('php://output');
        exit;
    }

    public function import()
    {
        $permissionCheck = $this->checkPermission('periode_audit_create'); // Asumsi izin import sama dengan create
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $file = $this->request->getFile('excel_file');
        if ($file && $file->isValid() && !$file->hasMoved()) {
            try {
                $spreadsheet = IOFactory::load($file->getTempName());
                $sheet = $spreadsheet->getActiveSheet();
                $dataRows = $sheet->toArray(null, true, true, true);

                if (count($dataRows) < 2) {
                    session()->setFlashdata('error', 'File Excel kosong atau hanya berisi header.');
                    return redirect()->to(base_url('/periode_audit'));
                }

                $headerRow = array_values(array_shift($dataRows));
                $dbFields = array_map(fn($h) => strtolower(str_replace(' ', '_', $h)), $headerRow);

                $importData = [];
                foreach ($dataRows as $row) {
                    if (empty(array_filter(array_values($row)))) continue; // Skip baris kosong
                    $rowData = array_combine($dbFields, array_values($row));
                    $importData[] = $rowData;
                }

                if (!empty($importData)) {
                    $this->Periode_auditModel->insertBatch($importData);
                    session()->setFlashdata('success', count($importData) . ' baris data berhasil diimpor.');
                } else {
                    session()->setFlashdata('error', 'Tidak ada data valid untuk diimpor.');
                }
            } catch (\Exception $e) {
                session()->setFlashdata('error', 'Gagal memproses file Excel: ' . $e->getMessage());
            }
        } else {
            session()->setFlashdata('error', 'Gagal mengunggah file. Pastikan file valid.');
        }

        return redirect()->to(base_url('/periode_audit'));
    }

    public function template()
    {
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Header kolom (hanya allowedFields)
        $headers = ['id_periode', 'tahun_audit', 'tgl_mulai', 'tgl_selesai', 'status_periode', 'created_by', 'updated_by'];
        $capitalizedHeaders = array_map(fn($h) => ucwords(str_replace('_', ' ', $h)), $headers);
        $sheet->fromArray($capitalizedHeaders, NULL, 'A1');

        $writer = new Xlsx($spreadsheet);
        $filename = 'periode_audit-template.xlsx';

        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header("Content-Disposition: attachment; filename=\"$filename\"");
        header('Cache-Control: max-age=0');
        $writer->save('php://output');
        exit;
    }
}
