<?php

namespace App\Controllers;

use App\Models\Penugasan_auditModel;
use App\Models\AuditorModel;
use App\Models\Periode_auditModel;
use App\Models\FakultaModel;
use App\Models\ProdiModel;
use App\Models\Unit_kerjaModel;
use CodeIgniter\Controller;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\IOFactory;
use App\Libraries\LogAktivitas;

class Penugasan_auditController extends BaseController
{
    protected $penugasanModel;
    protected $auditorModel;
    protected $periodeModel;
    protected $fakultasModel;
    protected $prodiModel;
    protected $unitKerjaModel;
    protected $logAktivitas;

    public function __construct()
    {
        $this->penugasanModel = new Penugasan_auditModel();
        $this->auditorModel  = new AuditorModel();
        $this->periodeModel  = new Periode_auditModel();
        $this->fakultasModel = new FakultaModel();
        $this->prodiModel    = new ProdiModel();
        $this->unitKerjaModel = new Unit_kerjaModel();
        $this->logAktivitas = new LogAktivitas();
    }

    public function index()
    {
        $permission = $this->checkPermission('penugasan_audit_view');
        if ($permission instanceof \CodeIgniter\HTTP\RedirectResponse) return $permission;

        $this->data['items'] = $this->penugasanModel->findAll();
        return view('penugasan_audit/index', $this->data);
    }

    /* ===================== CREATE ===================== */
    public function create()
    {
        $permission = $this->checkPermission('penugasan_audit_create');
        if ($permission instanceof \CodeIgniter\HTTP\RedirectResponse) return $permission;

        $this->data['auditors']   = $this->auditorModel->findAll();
        $this->data['periodes']   = $this->periodeModel->findAll();
        $this->data['fakultas']   = $this->fakultasModel->findAll();
        $this->data['prodi']      = $this->prodiModel->findAll();
        $this->data['unit_kerja'] = $this->unitKerjaModel->findAll();

        return view('penugasan_audit/create', $this->data);
    }

    /* ===================== STORE ===================== */
    public function store()
    {
        $permission = $this->checkPermission('penugasan_audit_create');
        if ($permission instanceof \CodeIgniter\HTTP\RedirectResponse) return $permission;

        $data = [
            'id_periode'    => $this->request->getPost('id_periode'),
            'id_auditor'    => $this->request->getPost('id_auditor'),
            'id_fakultas'   => $this->request->getPost('id_fakultas') ?: null,
            'id_prodi'      => $this->request->getPost('id_prodi') ?: null,
            'id_unit_kerja' => $this->request->getPost('id_unit_kerja') ?: null,
            'created_by'    => session()->get('id'),
        ];

        $this->penugasanModel->insert($data);
        $id = $this->penugasanModel->getInsertID();

        $keterangan = implode('<br>', [
            '<strong>Data yang ditambahkan:</strong>',
            '<strong>ID Penugasan</strong>: ' . $id,
            '<strong>ID Periode</strong>: ' . $data['id_periode'],
            '<strong>ID Auditor</strong>: ' . $data['id_auditor'],
            '<strong>ID Fakultas</strong>: ' . ($data['id_fakultas'] ?? '-'),
            '<strong>ID Prodi</strong>: ' . ($data['id_prodi'] ?? '-'),
            '<strong>ID Unit Kerja</strong>: ' . ($data['id_unit_kerja'] ?? '-'),
        ]);

        $user = $this->getUser();
        $this->logAktivitas->log(
            'create_penugasan_audit',
            $user['id'],
            $user['name'],
            current_url(),
            $keterangan
        );

        return redirect()->to('/penugasan_audit')->with('success', 'Data berhasil ditambahkan');
    }

    /* ===================== EDIT ===================== */
    public function edit($id)
    {
        $permission = $this->checkPermission('penugasan_audit_edit');
        if ($permission instanceof \CodeIgniter\HTTP\RedirectResponse) return $permission;

        $this->data['item']       = $this->penugasanModel->find($id);
        $this->data['auditors']   = $this->auditorModel->findAll();
        $this->data['periodes']   = $this->periodeModel->findAll();
        $this->data['fakultas']   = $this->fakultasModel->findAll();
        $this->data['prodi']      = $this->prodiModel->findAll();
        $this->data['unit_kerja'] = $this->unitKerjaModel->findAll();

        return view('penugasan_audit/edit', $this->data);
    }

    /* ===================== UPDATE (LOG PERUBAHAN) ===================== */
    public function update($id)
    {
        $permission = $this->checkPermission('penugasan_audit_edit');
        if ($permission instanceof \CodeIgniter\HTTP\RedirectResponse) return $permission;

        $lama = $this->penugasanModel->find($id);

        $baru = [
            'id_periode'    => $this->request->getPost('id_periode'),
            'id_auditor'    => $this->request->getPost('id_auditor'),
            'id_fakultas'   => $this->request->getPost('id_fakultas') ?: null,
            'id_prodi'      => $this->request->getPost('id_prodi') ?: null,
            'id_unit_kerja' => $this->request->getPost('id_unit_kerja') ?: null,
            'updated_by'    => session()->get('id'),
        ];

        $perubahan = [];
        foreach ($baru as $k => $v) {
            if (($lama[$k] ?? null) != $v) {
                $perubahan[] =
                    "<strong>{$k}</strong>:<br>Lama: \"{$lama[$k]}\"<br>Baru: \"{$v}\"";
            }
        }

        $keterangan = empty($perubahan)
            ? 'Menyimpan data tanpa perubahan'
            : implode('<hr>', $perubahan);

        $this->penugasanModel->update($id, $baru);

        $user = $this->getUser();
        $this->logAktivitas->log(
            'update_penugasan_audit',
            $user['id'],
            $user['name'],
            current_url(),
            $keterangan
        );

        return redirect()->to('/penugasan_audit')->with('success', 'Data berhasil diperbarui');
    }

    /* ===================== DELETE ===================== */
    public function delete($id)
    {
        $permission = $this->checkPermission('penugasan_audit_delete');
        if ($permission instanceof \CodeIgniter\HTTP\RedirectResponse) return $permission;

        $item = $this->penugasanModel->find($id);
        $this->penugasanModel->delete($id);

        $keterangan = implode('<br>', [
            '<strong>Data yang dihapus:</strong>',
            '<strong>ID Penugasan</strong>: ' . $id,
            '<strong>ID Periode</strong>: ' . $item['id_periode'],
            '<strong>ID Auditor</strong>: ' . $item['id_auditor'],
            '<strong>ID Fakultas</strong>: ' . ($item['id_fakultas'] ?? '-'),
            '<strong>ID Prodi</strong>: ' . ($item['id_prodi'] ?? '-'),
            '<strong>ID Unit Kerja</strong>: ' . ($item['id_unit_kerja'] ?? '-'),
        ]);

        $user = $this->getUser();
        $this->logAktivitas->log(
            'delete_penugasan_audit',
            $user['id'],
            $user['name'],
            current_url(),
            $keterangan
        );

        return redirect()->to('/penugasan_audit')->with('success', 'Data berhasil dihapus');
    }

    public function show($id)
    {
        $permissionCheck = $this->checkPermission('penugasan_audit_view');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data['item'] = $this->penugasanModel->find($id);
        return view('penugasan_audit/show', $this->data + $data);
    }
    private function getUser()
    {
        return [
            'id'   => session()->get('id'),
            'name' => session()->get('name'),
        ];
    }
    // --- FITUR EXCEL ---

    public function export()
    {
        $permissionCheck = $this->checkPermission('penugasan_audit_view'); // Asumsi izin export sama dengan view
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data = $this->penugasanModel->findAll();
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Header Kolom
        $headers = ['id', 'id_penugasan', 'id_periode', 'id_auditor', 'id_fakultas', 'id_prodi', 'id_unit_kerja', 'created_by', 'updated_by'];
        $capitalizedHeaders = array_map(fn($h) => ucwords(str_replace('_', ' ', $h)), $headers);
        $sheet->fromArray($capitalizedHeaders, NULL, 'A1');

        // Data
        $rowNumber = 2;
        foreach ($data as $item) {
            $rowData = [];
            foreach ($headers as $header) {
                $rowData[] = $item[$header] ?? '';
            }
            $sheet->fromArray($rowData, NULL, 'A' . $rowNumber++);
        }

        $writer = new Xlsx($spreadsheet);
        $filename = 'penugasan_audit-export-' . date('Y-m-d') . '.xlsx';

        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header("Content-Disposition: attachment; filename=\"$filename\"");
        header('Cache-Control: max-age=0');
        $writer->save('php://output');
        exit;
    }

    public function import()
    {
        $permissionCheck = $this->checkPermission('penugasan_audit_create'); // Asumsi izin import sama dengan create
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $file = $this->request->getFile('excel_file');
        if ($file && $file->isValid() && !$file->hasMoved()) {
            try {
                $spreadsheet = IOFactory::load($file->getTempName());
                $sheet = $spreadsheet->getActiveSheet();
                $dataRows = $sheet->toArray(null, true, true, true);

                if (count($dataRows) < 2) {
                    session()->setFlashdata('error', 'File Excel kosong atau hanya berisi header.');
                    return redirect()->to(base_url('/penugasan_audit'));
                }

                $headerRow = array_values(array_shift($dataRows));
                $dbFields = array_map(fn($h) => strtolower(str_replace(' ', '_', $h)), $headerRow);

                $importData = [];
                foreach ($dataRows as $row) {
                    if (empty(array_filter(array_values($row)))) continue; // Skip baris kosong
                    $rowData = array_combine($dbFields, array_values($row));
                    $importData[] = $rowData;
                }

                if (!empty($importData)) {
                    $this->penugasanModel->insertBatch($importData);
                    session()->setFlashdata('success', count($importData) . ' baris data berhasil diimpor.');
                } else {
                    session()->setFlashdata('error', 'Tidak ada data valid untuk diimpor.');
                }
            } catch (\Exception $e) {
                session()->setFlashdata('error', 'Gagal memproses file Excel: ' . $e->getMessage());
            }
        } else {
            session()->setFlashdata('error', 'Gagal mengunggah file. Pastikan file valid.');
        }

        return redirect()->to(base_url('/penugasan_audit'));
    }

    public function template()
    {
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Header kolom (hanya allowedFields)
        $headers = ['id_penugasan', 'id_periode', 'id_auditor', 'id_fakultas', 'id_prodi', 'id_unit_kerja', 'created_by', 'updated_by'];
        $capitalizedHeaders = array_map(fn($h) => ucwords(str_replace('_', ' ', $h)), $headers);
        $sheet->fromArray($capitalizedHeaders, NULL, 'A1');

        $writer = new Xlsx($spreadsheet);
        $filename = 'penugasan_audit-template.xlsx';

        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header("Content-Disposition: attachment; filename=\"$filename\"");
        header('Cache-Control: max-age=0');
        $writer->save('php://output');
        exit;
    }
}
