<?php

namespace App\Controllers;

use App\Models\Peluang_peningkatanModel;
use App\Models\Standar_lembaga_akreditasiModel;
use App\Models\Daftar_tilikModel;
use App\Models\FakultaModel;
use App\Models\ProdiModel;
use App\Models\Unit_kerjaModel;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\IOFactory;
use App\Libraries\LogAktivitas;


class Peluang_peningkatanController extends BaseController
{
    protected $peluangModel;
    protected $standarModel;
    protected $tilikModel;
    protected $fakultaModel;
    protected $prodiModel;
    protected $unitKerjaModel;
    protected $logAktivitas;


    public function __construct()
    {
        $this->peluangModel   = new Peluang_peningkatanModel();
        $this->standarModel   = new Standar_lembaga_akreditasiModel();
        $this->tilikModel     = new Daftar_tilikModel();
        $this->fakultaModel  = new FakultaModel();
        $this->prodiModel     = new ProdiModel();
        $this->unitKerjaModel = new Unit_kerjaModel();
        $this->logAktivitas = new LogAktivitas();
    }

    public function index()
    {
        if (!session()->get('isLoggedIn')) {
            return redirect()->to('/login');
        }

        $permissionCheck = $this->checkPermission('peluang_peningkatan_view');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data['items'] = $this->peluangModel
            ->select('
            peluang_peningkatan.*,
            creator.name AS created_by,
            updater.name AS updated_by,            s.nama_standar,
            t.pertanyaan,
            f.nama_fakultas,
            p.nama_prodi,
            u.nama_unit,
            creator.name AS created_by,
            updater.name AS updated_by
        ')
            ->join('standar_lembaga_akreditasi s', 's.id_standar = peluang_peningkatan.id_standar')
            ->join('daftar_tilik t', 't.id_tilik = peluang_peningkatan.id_tilik')
            ->join('fakultas f', 'f.id_fakultas = peluang_peningkatan.id_fakultas', 'left')
            ->join('prodi p', 'p.id_prodi = peluang_peningkatan.id_prodi', 'left')
            ->join('unit_kerja u', 'u.id_unit_kerja = peluang_peningkatan.id_unit_kerja', 'left')
            ->join('users AS creator', 'creator.id = peluang_peningkatan.created_by', 'left')
            ->join('users AS updater', 'updater.id = peluang_peningkatan.updated_by', 'left')
            ->findAll();

        return view('peluang_peningkatan/index', $this->data + $data);
    }


    public function create()
    {
        if (!session()->get('isLoggedIn')) {
            return redirect()->to('/login');
        }

        $permissionCheck = $this->checkPermission('peluang_peningkatan_create');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data = [
            'standar'     => (new Standar_lembaga_akreditasiModel())->findAll(),
            'tilik'       => (new Daftar_tilikModel())->findAll(),
            'fakultas'    => (new FakultaModel())->findAll(),
            'prodi'       => (new ProdiModel())->findAll(),
            'unit_kerja'  => (new Unit_kerjaModel())->findAll(),
        ];

        return view('peluang_peningkatan/create', $this->data + $data);
    }

    public function store()
    {
        $permissionCheck = $this->checkPermission('peluang_peningkatan_create');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data = [
            'id_standar'    => $this->request->getPost('id_standar'),
            'id_tilik'      => $this->request->getPost('id_tilik'),
            'id_fakultas'   => $this->request->getPost('id_fakultas') ?: null,
            'id_prodi'      => $this->request->getPost('id_prodi') ?: null,
            'id_unit_kerja' => $this->request->getPost('id_unit_kerja') ?: null,
            'bidang'        => $this->request->getPost('bidang'),
            'kelebihan'     => $this->request->getPost('kelebihan'),
            'peluang'       => $this->request->getPost('peluang'),
            'created_by'    => session()->get('id'),
        ];

        $this->peluangModel->insert($data);
        $id = $this->peluangModel->getInsertID();

        // 🔥 Ambil nama relasi
        $standar  = $this->standarModel->find($data['id_standar']);
        $tilik    = $this->tilikModel->find($data['id_tilik']);
        $fakultas = $data['id_fakultas'] ? $this->fakultaModel->find($data['id_fakultas']) : null;
        $prodi    = $data['id_prodi'] ? $this->prodiModel->find($data['id_prodi']) : null;
        $unit     = $data['id_unit_kerja'] ? $this->unitKerjaModel->find($data['id_unit_kerja']) : null;

        // ✅ KETERANGAN LENGKAP
        $keterangan = implode('<br>', [
            '<strong>Data yang ditambahkan:</strong>',
            '<strong>ID</strong>: ' . $id,
            '<strong>Standar</strong>: ' . esc($standar['nama_standar'] ?? '-'),
            '<strong>Daftar Tilik</strong>: ' . esc($tilik['pertanyaan'] ?? '-'),
            '<strong>Fakultas</strong>: ' . esc($fakultas['nama_fakultas'] ?? '-'),
            '<strong>Prodi</strong>: ' . esc($prodi['nama_prodi'] ?? '-'),
            '<strong>Unit Kerja</strong>: ' . esc($unit['nama_unit'] ?? '-'),
            '<strong>Bidang</strong>: ' . esc($data['bidang']),
            '<strong>Kelebihan</strong>:<br>' . nl2br(esc($data['kelebihan'])),
            '<strong>Peluang</strong>:<br>' . nl2br(esc($data['peluang'])),
        ]);

        $user = $this->getUser();
        $this->logAktivitas->log(
            'create_peluang_peningkatan',
            $user['id'],
            $user['name'],
            current_url(),
            $keterangan
        );

        return redirect()->to('/peluang_peningkatan')
            ->with('success', 'Data berhasil ditambahkan');
    }



    public function edit($id)
    {
        if (!session()->get('isLoggedIn')) {
            return redirect()->to('/login');
        }

        $permissionCheck = $this->checkPermission('peluang_peningkatan_edit');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data = [
            'item'     => $this->peluangModel->find($id),
            'standar'  => $this->standarModel->findAll(),
            'tilik'    => $this->tilikModel->findAll(),
            'fakultas' => $this->fakultaModel->findAll(),
            'prodi'    => $this->prodiModel->findAll(),
            'unit'     => $this->unitKerjaModel->findAll(),
        ];

        return view('peluang_peningkatan/edit', $this->data + $data);
    }
    public function update($id)
    {
        if (!session()->get('isLoggedIn')) {
            return redirect()->to('/login');
        }

        $permissionCheck = $this->checkPermission('peluang_peningkatan_edit');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $lama = $this->peluangModel->find($id);
        if (!$lama) {
            return redirect()->to('/peluang_peningkatan')->with('error', 'Data tidak ditemukan');
        }

        $post = $this->request->getPost();

        $fields = [
            'bidang'    => 'Bidang',
            'kelebihan' => 'Kelebihan',
            'peluang'   => 'Peluang',
        ];

        $perubahan = [];

        foreach ($fields as $field => $label) {
            if (
                isset($post[$field]) &&
                $lama[$field] !== $post[$field]
            ) {
                $perubahan[] =
                    "<strong>{$label}</strong>:<br>" .
                    "Lama: \"{$lama[$field]}\"<br>" .
                    "Baru: \"{$post[$field]}\"";
            }
        }

        $keterangan = empty($perubahan)
            ? 'Menyimpan data tanpa perubahan nilai'
            : implode('<hr>', $perubahan);

        $post['updated_by'] = session()->get('id');
        $this->peluangModel->update($id, $post);

        $user = $this->getUser();
        $this->logAktivitas->log(
            'update_peluang_peningkatan',
            $user['id'],
            $user['name'],
            current_url(),
            $keterangan
        );

        return redirect()->to('/peluang_peningkatan')
            ->with('success', 'Data berhasil diperbarui');
    }

    public function delete($id)
    {
        if (!session()->get('isLoggedIn')) {
            return redirect()->to('/login');
        }

        $permissionCheck = $this->checkPermission('peluang_peningkatan_delete');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $item = $this->peluangModel->find($id);
        if (!$item) {
            return redirect()->to('/peluang_peningkatan')->with('error', 'Data tidak ditemukan');
        }

        $standar = $this->standarModel->find($item['id_standar']);
        $tilik   = $this->tilikModel->find($item['id_tilik']);

        $this->peluangModel->delete($id);

        $keterangan = implode('<br>', [
            '<strong>Data yang dihapus:</strong>',
            '<strong>ID</strong>: ' . $id,
            '<strong>Standar</strong>: ' . esc($standar['nama_standar'] ?? '-'),
            '<strong>Daftar Tilik</strong>: ' . esc($tilik['pertanyaan'] ?? '-'),
            '<strong>Bidang</strong>: ' . esc($item['bidang']),
        ]);

        $user = $this->getUser();
        $this->logAktivitas->log(
            'delete_peluang_peningkatan',
            $user['id'],
            $user['name'],
            current_url(),
            $keterangan
        );

        return redirect()->to('/peluang_peningkatan')
            ->with('success', 'Data berhasil dihapus');
    }


    public function show($id)
    {
        $permissionCheck = $this->checkPermission('peluang_peningkatan_view');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data['item'] = $this->peluangModel->find($id);
        return view('peluang_peningkatan/show', $data);
    }
    private function getUser()
    {
        return [
            'id'   => session()->get('id'),
            'name' => session()->get('name'),
        ];
    }
    // --- FITUR EXCEL ---

    public function export()
    {
        $permissionCheck = $this->checkPermission('peluang_peningkatan_view'); // Asumsi izin export sama dengan view
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data = $this->peluangModel->findAll();
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Header Kolom
        $headers = ['id', 'id_peluang', 'id_standar', 'id_tilik', 'id_fakultas', 'id_prodi', 'id_unit_kerja', 'bidang', 'kelebihan', 'peluang', 'created_by', 'updated_by'];
        $capitalizedHeaders = array_map(fn($h) => ucwords(str_replace('_', ' ', $h)), $headers);
        $sheet->fromArray($capitalizedHeaders, NULL, 'A1');

        // Data
        $rowNumber = 2;
        foreach ($data as $item) {
            $rowData = [];
            foreach ($headers as $header) {
                $rowData[] = $item[$header] ?? '';
            }
            $sheet->fromArray($rowData, NULL, 'A' . $rowNumber++);
        }

        $writer = new Xlsx($spreadsheet);
        $filename = 'peluang_peningkatan-export-' . date('Y-m-d') . '.xlsx';

        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header("Content-Disposition: attachment; filename=\"$filename\"");
        header('Cache-Control: max-age=0');
        $writer->save('php://output');
        exit;
    }

    public function import()
    {
        $permissionCheck = $this->checkPermission('peluang_peningkatan_create'); // Asumsi izin import sama dengan create
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $file = $this->request->getFile('excel_file');
        if ($file && $file->isValid() && !$file->hasMoved()) {
            try {
                $spreadsheet = IOFactory::load($file->getTempName());
                $sheet = $spreadsheet->getActiveSheet();
                $dataRows = $sheet->toArray(null, true, true, true);

                if (count($dataRows) < 2) {
                    session()->setFlashdata('error', 'File Excel kosong atau hanya berisi header.');
                    return redirect()->to(base_url('/peluang_peningkatan'));
                }

                $headerRow = array_values(array_shift($dataRows));
                $dbFields = array_map(fn($h) => strtolower(str_replace(' ', '_', $h)), $headerRow);

                $importData = [];
                foreach ($dataRows as $row) {
                    if (empty(array_filter(array_values($row)))) continue; // Skip baris kosong
                    $rowData = array_combine($dbFields, array_values($row));
                    $importData[] = $rowData;
                }

                if (!empty($importData)) {
                    $this->peluangModel->insertBatch($importData);
                    session()->setFlashdata('success', count($importData) . ' baris data berhasil diimpor.');
                } else {
                    session()->setFlashdata('error', 'Tidak ada data valid untuk diimpor.');
                }
            } catch (\Exception $e) {
                session()->setFlashdata('error', 'Gagal memproses file Excel: ' . $e->getMessage());
            }
        } else {
            session()->setFlashdata('error', 'Gagal mengunggah file. Pastikan file valid.');
        }

        return redirect()->to(base_url('/peluang_peningkatan'));
    }

    public function template()
    {
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Header kolom (hanya allowedFields)
        $headers = ['id_peluang', 'id_standar', 'id_tilik', 'id_fakultas', 'id_prodi', 'id_unit_kerja', 'bidang', 'kelebihan', 'peluang', 'created_by', 'updated_by'];
        $capitalizedHeaders = array_map(fn($h) => ucwords(str_replace('_', ' ', $h)), $headers);
        $sheet->fromArray($capitalizedHeaders, NULL, 'A1');

        $writer = new Xlsx($spreadsheet);
        $filename = 'peluang_peningkatan-template.xlsx';

        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header("Content-Disposition: attachment; filename=\"$filename\"");
        header('Cache-Control: max-age=0');
        $writer->save('php://output');
        exit;
    }
}
