<?php

namespace App\Controllers;

use App\Models\MenuModel;
use CodeIgniter\Controller;
use App\Libraries\LogAktivitas;

class Menu extends BaseController
{
    protected $menuModel;
    protected $logAktivitas;

    public function __construct()
    {
        parent::initController(
            service('request'),
            service('response'),
            service('logger')
        );

        $this->menuModel = new MenuModel();
        $this->logAktivitas = new LogAktivitas();
        helper('form');
    }

    // List all menus, access controlled by permission 'menu_view'
    public function index()
    {
        //$this->checkPermission('menu_view');
        $permissionCheck = $this->checkPermission('menu_view');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck; // langsung redirect jika tidak punya izin
        }
        $data = $this->data;
        $data['title'] = 'Menu Management';
        // Fetch all menus without filtering for management purposes
        $data['menus'] = $this->menuModel->orderBy('sort_order', 'ASC')->findAll();
        // var_dump($data);die;

        return view('menu/index', $data);
    }

    // Show create menu form, access controlled by 'menu_create'
    public function create()
    {
        //$this->checkPermission('menu_create');
        $permissionCheck = $this->checkPermission('menu_create');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck; // langsung redirect jika tidak punya izin
        }

        $data = $this->data;
        $data['title'] = 'Create Menu';

        // Fetch all menus for selecting parent menu
        $data['parentMenus'] = $this->menuModel->orderBy('sort_order', 'ASC')->findAll();

        return view('menu/create', $data);
    }

    // Store new menu, access controlled by 'menu_create'
    public function store()
    {
        //$this->checkPermission('menu_create');

        $permissionCheck = $this->checkPermission('menu_create');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck; // langsung redirect jika tidak punya izin
        }

        $rules = [
            'title'      => 'required|min_length[3]|max_length[255]',
            'url'        => 'permit_empty|max_length[255]',
            'icon'       => 'permit_empty|max_length[100]',
            'parent_id'  => 'permit_empty|numeric',
            'permission' => 'permit_empty|max_length[255]',
            'sort_order' => 'required|integer',
        ];

        if (!$this->validate($rules)) {
            $data = $this->data;
            $data['validation'] = $this->validator;
            $data['title'] = 'Create Menu';
            $data['parentMenus'] = $this->menuModel->orderBy('sort_order', 'ASC')->findAll();

            return view('menu/create', $data);
        }

        $menuData = [
            'title'      => $this->request->getPost('title'),
            'url'        => $this->request->getPost('url') ?: '#',
            'icon'       => $this->request->getPost('icon') ?: 'fa-circle',
            'parent_id'  => $this->request->getPost('parent_id') ?: null,
            'permission' => $this->request->getPost('permission'),
            'sort_order' => (int)$this->request->getPost('sort_order'),
        ];

        $this->menuModel->save($menuData);

        // Log menu creation
        $this->logAktivitas->log('create_menu', $this->data['user']['id'], $this->data['user']['name'], current_url());

        session()->setFlashdata('success', 'Menu created successfully.');
        return redirect()->to('/menus');
    }

    // Show edit form, access controlled by 'menu_edit'
    public function edit($id = null)
    {
        //$this->checkPermission('menu_edit');
        $permissionCheck = $this->checkPermission('menu_edit');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck; // langsung redirect jika tidak punya izin
        }

        if (!$id || !$menu = $this->menuModel->find($id)) {
            throw \CodeIgniter\Exceptions\PageNotFoundException::forPageNotFound('Menu not found');
        }

        $data = $this->data;
        $data['title'] = 'Edit Menu';
        $data['menu'] = $menu;
        $data['parentMenus'] = $this->menuModel->where('id !=', $id)->orderBy('sort_order', 'ASC')->findAll();

        return view('menu/edit', $data);
    }

    // Update menu, access controlled by 'menu_edit'
    public function update($id = null)
    {
        //$this->checkPermission('menu_edit');
        $permissionCheck = $this->checkPermission('menu_edit');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck; // langsung redirect jika tidak punya izin
        }

        if (!$id || !$this->menuModel->find($id)) {
            throw \CodeIgniter\Exceptions\PageNotFoundException::forPageNotFound('Menu not found');
        }

        $rules = [
            'title'      => 'required|min_length[3]|max_length[255]',
            'url'        => 'permit_empty|max_length[255]',
            'icon'       => 'permit_empty|max_length[100]',
            'parent_id'  => 'permit_empty|numeric|not_in_list[' . $id . ']',
            'permission' => 'permit_empty|max_length[255]',
            'sort_order' => 'required|integer',
        ];

        if (!$this->validate($rules)) {
            $data = $this->data;
            $data['validation'] = $this->validator;
            $data['title'] = 'Edit Menu';
            $data['menu'] = $this->menuModel->find($id);
            $data['parentMenus'] = $this->menuModel->where('id !=', $id)->orderBy('sort_order', 'ASC')->findAll();

            return view('menu/edit', $data);
        }

        $menuData = [
            'id'         => $id,
            'title'      => $this->request->getPost('title'),
            'url'        => $this->request->getPost('url') ?: '#',
            'icon'       => $this->request->getPost('icon') ?: 'fa-circle',
            'parent_id'  => $this->request->getPost('parent_id') ?: null,
            'permission' => $this->request->getPost('permission'),
            'sort_order' => (int)$this->request->getPost('sort_order'),
        ];

        $this->menuModel->save($menuData);

        // Log menu update
        $this->logAktivitas->log('update_menu', $this->data['user']['id'], $this->data['user']['name'], current_url());

        session()->setFlashdata('success', 'Menu updated successfully.');
        return redirect()->to('/menus');
    }

    // Delete menu, access controlled by 'menu_delete'
    public function delete($id = null)
    {
        //$this->checkPermission('menu_delete');
        $permissionCheck = $this->checkPermission('menu_delete');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck; // langsung redirect jika tidak punya izin
        }
        if (!$id || !$this->menuModel->find($id)) {
            throw \CodeIgniter\Exceptions\PageNotFoundException::forPageNotFound('Menu not found');
        }

        $this->menuModel->delete($id);

        // Log menu deletion
        $this->logAktivitas->log('delete_menu', $this->data['user']['id'], $this->data['user']['name'], current_url());

        session()->setFlashdata('success', 'Menu deleted successfully.');
        return redirect()->to('/menus');
    }
}
