<?php

namespace App\Controllers;

use App\Models\Matrix_penilaianModel;
use App\Models\Standar_lembaga_akreditasiModel;
use CodeIgniter\Controller;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\IOFactory;
use App\Libraries\LogAktivitas;

class Matrix_penilaianController extends BaseController
{
    protected $matrixModel;
    protected $standarModel;
    protected $logAktivitas;


    public function __construct()
    {
        $this->matrixModel  = new Matrix_penilaianModel();
        $this->standarModel = new Standar_lembaga_akreditasiModel();
        $this->logAktivitas = new LogAktivitas();
    }

    public function index()
    {
        if (!session()->get('isLoggedIn')) {
            return redirect()->to('/login');
        }

        $permissionCheck = $this->checkPermission('matrix_penilaian_view');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data['items'] = $this->matrixModel
            ->select('matrix_penilaian.*, standar_lembaga_akreditasi.nama_standar')
            ->join(
                'standar_lembaga_akreditasi',
                'standar_lembaga_akreditasi.id_standar = matrix_penilaian.id_standar'
            )
            ->findAll();

        return view('matrix_penilaian/index', $this->data + $data);
    }


    public function create()
    {
        if (!session()->get('isLoggedIn')) {
            return redirect()->to('/login');
        }

        $permissionCheck = $this->checkPermission('matrix_penilaian_create');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data['standar'] = $this->standarModel->findAll();
        return view('matrix_penilaian/create', $this->data + $data);
    }

    public function store()
    {
        if (!session()->get('isLoggedIn')) {
            return redirect()->to('/login');
        }

        $permissionCheck = $this->checkPermission('matrix_penilaian_create');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data = [
            'id_standar'         => $this->request->getPost('id_standar'),
            'deskripsi_kriteria' => $this->request->getPost('deskripsi_kriteria'),
            'nilai'              => $this->request->getPost('nilai'),
            'created_by'         => session()->get('id'),
        ];

        $this->matrixModel->insert($data);

        // ID ASLI DATABASE
        $idBaru = $this->matrixModel->getInsertID();

        // Ambil nama standar
        $standar = $this->standarModel->find($data['id_standar']);

        // KETERANGAN LOG
        $keterangan = [];
        $keterangan[] = '<strong>Data yang ditambahkan:</strong>';
        $keterangan[] = '<strong>ID Matrix</strong>: ' . $idBaru;
        $keterangan[] = '<strong>Standar</strong>: ' . ($standar['nama_standar'] ?? '-');
        $keterangan[] = '<strong>Deskripsi Kriteria</strong>: ' . esc($data['deskripsi_kriteria']);
        $keterangan[] = '<strong>Nilai</strong>: ' . esc($data['nilai']);

        $user = $this->getUser();
        $this->logAktivitas->log(
            'create_matrix_penilaian',
            $user['id'],
            $user['name'],
            current_url(),
            implode('<br>', $keterangan)
        );

        return redirect()->to('/matrix_penilaian')->with('success', 'Matrix penilaian berhasil ditambahkan');
    }


    public function edit($id)
    {
        if (!session()->get('isLoggedIn')) {
            return redirect()->to('/login');
        }

        $permissionCheck = $this->checkPermission('matrix_penilaian_edit');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data = [
            'item'    => $this->matrixModel->find($id),
            'standar' => $this->standarModel->findAll()
        ];

        return view('matrix_penilaian/edit', $this->data + $data);
    }

    public function update($id)
    {
        if (!session()->get('isLoggedIn')) {
            return redirect()->to('/login');
        }

        $permissionCheck = $this->checkPermission('matrix_penilaian_edit');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        // DATA LAMA
        $old = $this->matrixModel->find($id);

        $data = [
            'id_standar'         => $this->request->getPost('id_standar'),
            'deskripsi_kriteria' => $this->request->getPost('deskripsi_kriteria'),
            'nilai'              => $this->request->getPost('nilai'),
            'updated_by'         => session()->get('id'),
        ];

        $this->matrixModel->update($id, $data);

        $standarLama = $this->standarModel->find($old['id_standar']);
        $standarBaru = $this->standarModel->find($data['id_standar']);

        // KETERANGAN LOG
        $keterangan = [];
        $keterangan[] = '<strong>Data yang diperbarui:</strong>';
        $keterangan[] = '<strong>ID Matrix</strong>: ' . $id;

        if ($old['id_standar'] != $data['id_standar']) {
            $keterangan[] =
                '<strong>Standar</strong>:<br>' .
                'Sebelum: ' . ($standarLama['nama_standar'] ?? '-') . '<br>' .
                'Sesudah: ' . ($standarBaru['nama_standar'] ?? '-');
        }

        if ($old['deskripsi_kriteria'] !== $data['deskripsi_kriteria']) {
            $keterangan[] =
                '<strong>Deskripsi Kriteria</strong>:<br>' .
                'Sebelum: "' . esc($old['deskripsi_kriteria']) . '"<br>' .
                'Sesudah: "' . esc($data['deskripsi_kriteria']) . '"';
        }

        if ($old['nilai'] !== $data['nilai']) {
            $keterangan[] =
                '<strong>Nilai</strong>: ' .
                $old['nilai'] . ' → ' . $data['nilai'];
        }

        $user = $this->getUser();
        $this->logAktivitas->log(
            'update_matrix_penilaian',
            $user['id'],
            $user['name'],
            current_url(),
            implode('<hr>', $keterangan)
        );

        return redirect()->to('/matrix_penilaian')->with('success', 'Data berhasil diperbarui');
    }
    public function delete($id)
    {
        if (!session()->get('isLoggedIn')) {
            return redirect()->to('/login');
        }

        $permissionCheck = $this->checkPermission('matrix_penilaian_delete');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        // DATA SEBELUM HAPUS
        $item = $this->matrixModel->find($id);
        $standar = $this->standarModel->find($item['id_standar']);

        $this->matrixModel->delete($id);

        // KETERANGAN LOG
        $keterangan = [];
        $keterangan[] = '<strong>Data yang dihapus:</strong>';
        $keterangan[] = '<strong>ID Matrix</strong>: ' . $id;
        $keterangan[] = '<strong>Standar</strong>: ' . ($standar['nama_standar'] ?? '-');
        $keterangan[] = '<strong>Deskripsi Kriteria</strong>: ' . esc($item['deskripsi_kriteria']);
        $keterangan[] = '<strong>Nilai</strong>: ' . esc($item['nilai']);

        $user = $this->getUser();
        $this->logAktivitas->log(
            'delete_matrix_penilaian',
            $user['id'],
            $user['name'],
            current_url(),
            implode('<br>', $keterangan)
        );

        return redirect()->to('/matrix_penilaian')->with('success', 'Data berhasil dihapus');
    }


    public function show($id)
    {
        $permissionCheck = $this->checkPermission('matrix_penilaian_view');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data['item'] = $this->matrixModel->find($id);
        return view('matrix_penilaian/show', $this->data + $data);
    }

    private function getUser()
    {
        return [
            'id'   => session()->get('id'),
            'name' => session()->get('name'),
        ];
    }
    // --- FITUR EXCEL ---

    public function export()
    {
        $permissionCheck = $this->checkPermission('matrix_penilaian_view'); // Asumsi izin export sama dengan view
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data = $this->matrixModel->findAll();
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Header Kolom
        $headers = ['id', 'id_matrix', 'id_standar', 'deskripsi_kriteria', 'nilai', 'created_by', 'updated_by'];
        $capitalizedHeaders = array_map(fn($h) => ucwords(str_replace('_', ' ', $h)), $headers);
        $sheet->fromArray($capitalizedHeaders, NULL, 'A1');

        // Data
        $rowNumber = 2;
        foreach ($data as $item) {
            $rowData = [];
            foreach ($headers as $header) {
                $rowData[] = $item[$header] ?? '';
            }
            $sheet->fromArray($rowData, NULL, 'A' . $rowNumber++);
        }

        $writer = new Xlsx($spreadsheet);
        $filename = 'matrix_penilaian-export-' . date('Y-m-d') . '.xlsx';

        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header("Content-Disposition: attachment; filename=\"$filename\"");
        header('Cache-Control: max-age=0');
        $writer->save('php://output');
        exit;
    }

    public function import()
    {
        $permissionCheck = $this->checkPermission('matrix_penilaian_create'); // Asumsi izin import sama dengan create
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $file = $this->request->getFile('excel_file');
        if ($file && $file->isValid() && !$file->hasMoved()) {
            try {
                $spreadsheet = IOFactory::load($file->getTempName());
                $sheet = $spreadsheet->getActiveSheet();
                $dataRows = $sheet->toArray(null, true, true, true);

                if (count($dataRows) < 2) {
                    session()->setFlashdata('error', 'File Excel kosong atau hanya berisi header.');
                    return redirect()->to(base_url('/matrix_penilaian'));
                }

                $headerRow = array_values(array_shift($dataRows));
                $dbFields = array_map(fn($h) => strtolower(str_replace(' ', '_', $h)), $headerRow);

                $importData = [];
                foreach ($dataRows as $row) {
                    if (empty(array_filter(array_values($row)))) continue; // Skip baris kosong
                    $rowData = array_combine($dbFields, array_values($row));
                    $importData[] = $rowData;
                }

                if (!empty($importData)) {
                    $this->matrixModel->insertBatch($importData);
                    session()->setFlashdata('success', count($importData) . ' baris data berhasil diimpor.');
                } else {
                    session()->setFlashdata('error', 'Tidak ada data valid untuk diimpor.');
                }
            } catch (\Exception $e) {
                session()->setFlashdata('error', 'Gagal memproses file Excel: ' . $e->getMessage());
            }
        } else {
            session()->setFlashdata('error', 'Gagal mengunggah file. Pastikan file valid.');
        }

        return redirect()->to(base_url('/matrix_penilaian'));
    }

    public function template()
    {
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Header kolom (hanya allowedFields)
        $headers = ['id_matrix', 'id_standar', 'deskripsi_kriteria', 'nilai', 'created_by', 'updated_by'];
        $capitalizedHeaders = array_map(fn($h) => ucwords(str_replace('_', ' ', $h)), $headers);
        $sheet->fromArray($capitalizedHeaders, NULL, 'A1');

        $writer = new Xlsx($spreadsheet);
        $filename = 'matrix_penilaian-template.xlsx';

        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header("Content-Disposition: attachment; filename=\"$filename\"");
        header('Cache-Control: max-age=0');
        $writer->save('php://output');
        exit;
    }
}
