<?php

namespace App\Controllers;

use App\Models\LogAktivitasModel;
use CodeIgniter\Controller;

class LogAktivitasController extends BaseController
{
    protected $logAktivitasModel;

    public function __construct()
    {
        parent::initController(
            service('request'),
            service('response'),
            service('logger')
        );
        helper(['form', 'url']);
        $this->logAktivitasModel = new LogAktivitasModel();
    }

    public function index()
    {
        $permissionCheck = $this->checkPermission('log_aktivitas_view');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $page = $this->request->getGet('page') ?? 1;
        $perPage = 20;
        $offset = ($page - 1) * $perPage;

        $logs = $this->logAktivitasModel->getLogs($perPage, $offset);
        $totalLogs = $this->logAktivitasModel->countLogs();
        $totalPages = ceil($totalLogs / $perPage);

        $data = array_merge($this->data, [
            'logs' => $logs,
            'currentPage' => $page,
            'totalPages' => $totalPages,
            'totalLogs' => $totalLogs,
            'perPage' => $perPage,
            'menus' => $this->data['menus'],
            'user' => $this->data['user']
        ]);

        return view('log_aktivitas/index', $data);
    }

    public function filter()
    {
        $permissionCheck = $this->checkPermission('log_aktivitas_view');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $action = $this->request->getGet('action');
        $userId = $this->request->getGet('user_id');
        $dateFrom = $this->request->getGet('date_from');
        $dateTo = $this->request->getGet('date_to');
        $page = $this->request->getGet('page') ?? 1;
        $perPage = 20;
        $offset = ($page - 1) * $perPage;

        $query = $this->logAktivitasModel;

        if (!empty($action)) {
            $query = $query->where('aksi', $action);
        }

        if (!empty($userId)) {
            $query = $query->where('user_id', $userId);
        }

        if (!empty($dateFrom)) {
            $query = $query->where('created_at >=', $dateFrom . ' 00:00:00');
        }

        if (!empty($dateTo)) {
            $query = $query->where('created_at <=', $dateTo . ' 23:59:59');
        }

        $logs = $query->orderBy('created_at', 'DESC')->findAll($perPage, $offset);
        $totalLogs = $query->countAllResults(false); // false to not reset query
        $totalPages = ceil($totalLogs / $perPage);

        $data = array_merge($this->data, [
            'logs' => $logs,
            'currentPage' => $page,
            'totalPages' => $totalPages,
            'totalLogs' => $totalLogs,
            'perPage' => $perPage,
            'filters' => [
                'aksi' => $action,
                'user_id' => $userId,
                'date_from' => $dateFrom,
                'date_to' => $dateTo
            ],
            'menus' => $this->data['menus'],
            'user' => $this->data['user']
        ]);

        return view('log_aktivitas/index', $data);
    }

    public function show($id)
    {
        $permissionCheck = $this->checkPermission('log_aktivitas_view');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $log = $this->logAktivitasModel->find($id);
        if (!$log) {
            throw \CodeIgniter\Exceptions\PageNotFoundException::forPageNotFound('Log entry not found');
        }

        $data = array_merge($this->data, [
            'log' => $log,
            'menus' => $this->data['menus'],
            'user' => $this->data['user']
        ]);

        return view('log_aktivitas/show', $data);
    }

    // API endpoint for AJAX requests (optional)
    public function api_logs()
    {
        $permissionCheck = $this->checkPermission('log_aktivitas_view');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $this->response->setJSON(['error' => 'Unauthorized'])->setStatusCode(403);
        }

        $page = $this->request->getGet('page') ?? 1;
        $perPage = $this->request->getGet('per_page') ?? 20;
        $offset = ($page - 1) * $perPage;

        $logs = $this->logAktivitasModel->getLogs($perPage, $offset);
        $totalLogs = $this->logAktivitasModel->countLogs();

        return $this->response->setJSON([
            'logs' => $logs,
            'pagination' => [
                'current_page' => (int)$page,
                'per_page' => (int)$perPage,
                'total' => $totalLogs,
                'total_pages' => ceil($totalLogs / $perPage)
            ]
        ]);
    }
}
