<?php

namespace App\Controllers;

use App\Models\Lemb_akreditasiModel;
use CodeIgniter\Controller;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\IOFactory;
use App\Libraries\LogAktivitas;

class Lemb_akreditasiController extends BaseController
{
    protected $Lemb_akreditasiModel;
    protected $logAktivitas;

    public function __construct()
    {
        $this->Lemb_akreditasiModel = new Lemb_akreditasiModel();
        $this->logAktivitas = new LogAktivitas();
    }

    public function index()
    {
        if (!session()->get('isLoggedIn')) {
            return redirect()->to('/login');
        }

        $permissionCheck = $this->checkPermission('lemb_akreditasi_view');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data['items'] = $this->Lemb_akreditasiModel->findAll();
        return view('lemb_akreditasi/index', $this->data + $data);
    }

    public function create()
    {
        $permissionCheck = $this->checkPermission('lemb_akreditasi_create');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        return view('lemb_akreditasi/create', $this->data);
    }

    public function store()
    {
        $permissionCheck = $this->checkPermission('lemb_akreditasi_create');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data = [
            'nama_lembaga' => $this->request->getPost('nama_lembaga'),
            'created_by'   => session()->get('id'),
            'updated_by'   => session()->get('id')
        ];

        // INSERT
        $this->Lemb_akreditasiModel->insert($data);

        // AMBIL ID ASLI DATABASE
        $idBaru = $this->Lemb_akreditasiModel->getInsertID();

        // KETERANGAN LOG
        $keterangan = [];
        $keterangan[] = '<strong>Data yang ditambahkan:</strong>';
        $keterangan[] = '<strong>ID</strong>: ' . $idBaru;
        $keterangan[] = '<strong>Nama Lembaga</strong>: ' . esc($data['nama_lembaga']);

        $keteranganLog = implode('<br>', $keterangan);

        $user = $this->getUser();
        $this->logAktivitas->log(
            'create_lemb_akreditasi',
            $user['id'],
            $user['name'],
            current_url(),
            $keteranganLog
        );

        return redirect()->to('/lemb_akreditasi')->with('success', 'Data berhasil ditambahkan.');
    }


    public function edit($id)
    {
        $permissionCheck = $this->checkPermission('lemb_akreditasi_edit');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data['item'] = $this->Lemb_akreditasiModel->find($id);
        $user = $this->getUser();
        $this->logAktivitas->log('edit_lemb_akreditasi', $user['id'], $user['name'], current_url());

        return view('lemb_akreditasi/edit', $this->data + $data);
    }

    public function update($id)
    {
        $permissionCheck = $this->checkPermission('lemb_akreditasi_edit');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        // DATA LAMA
        $old = $this->Lemb_akreditasiModel->find($id);

        $data = [
            'nama_lembaga' => $this->request->getPost('nama_lembaga'),
            'updated_by'   => session()->get('id')
        ];

        $this->Lemb_akreditasiModel->update($id, $data);

        // KETERANGAN
        $keterangan = [];
        $keterangan[] = '<strong>Data yang diperbarui:</strong>';
        $keterangan[] = '<strong>ID</strong>: ' . $id;

        if ($old['nama_lembaga'] !== $data['nama_lembaga']) {
            $keterangan[] =
                '<strong>Nama Lembaga</strong>:<br>' .
                'Sebelum: "' . esc($old['nama_lembaga']) . '"<br>' .
                'Sesudah: "' . esc($data['nama_lembaga']) . '"';
        }

        $keteranganLog = implode('<hr>', $keterangan);

        $user = $this->getUser();
        $this->logAktivitas->log(
            'update_lemb_akreditasi',
            $user['id'],
            $user['name'],
            current_url(),
            $keteranganLog
        );

        return redirect()->to('/lemb_akreditasi')->with('success', 'Data berhasil diperbarui.');
    }


    public function delete($id)
    {
        $permissionCheck = $this->checkPermission('lemb_akreditasi_delete');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        // DATA SEBELUM DIHAPUS
        $item = $this->Lemb_akreditasiModel->find($id);

        $this->Lemb_akreditasiModel->delete($id);

        // KETERANGAN
        $keterangan = [];
        $keterangan[] = '<strong>Data yang dihapus:</strong>';
        $keterangan[] = '<strong>ID</strong>: ' . $id;
        $keterangan[] = '<strong>Nama Lembaga</strong>: ' . esc($item['nama_lembaga']);

        $keteranganLog = implode('<br>', $keterangan);

        $user = $this->getUser();
        $this->logAktivitas->log(
            'delete_lemb_akreditasi',
            $user['id'],
            $user['name'],
            current_url(),
            $keteranganLog
        );

        return redirect()->to('/lemb_akreditasi')->with('success', 'Data berhasil dihapus.');
    }


    public function show($id)
    {
        $permissionCheck = $this->checkPermission('lemb_akreditasi_view');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data['item'] = $this->Lemb_akreditasiModel->find($id);
        return view('lemb_akreditasi/show', $this->data + $data);
    }

    private function getUser()
    {
        return [
            'id'   => session()->get('id'),
            'name' => session()->get('name')
        ];
    }

    // -----------------------------
    // EXPORT EXCEL
    // -----------------------------
    public function export()
    {
        $permissionCheck = $this->checkPermission('lemb_akreditasi_view');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data = $this->Lemb_akreditasiModel->findAll();
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        $headers = ['ID', 'Nama Lembaga', 'Created By', 'Updated By', 'Created At', 'Updated At'];
        $sheet->fromArray($headers, NULL, 'A1');

        $row = 2;
        foreach ($data as $item) {
            $sheet->setCellValue("A$row", $item['id_lemb']);
            $sheet->setCellValue("B$row", $item['nama_lembaga']);
            $sheet->setCellValue("C$row", $item['created_by']);
            $sheet->setCellValue("D$row", $item['updated_by']);
            $sheet->setCellValue("E$row", $item['created_at']);
            $sheet->setCellValue("F$row", $item['updated_at']);
            $row++;
        }

        $writer = new Xlsx($spreadsheet);
        $filename = 'lemb_akreditasi-' . date('Y-m-d') . '.xlsx';

        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header("Content-Disposition: attachment; filename=\"$filename\"");
        header('Cache-Control: max-age=0');
        $writer->save('php://output');
        exit;
    }

    // -----------------------------
    // IMPORT EXCEL
    // -----------------------------
    public function import()
    {
        $permissionCheck = $this->checkPermission('lemb_akreditasi_create');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $file = $this->request->getFile('excel_file');
        if (!$file->isValid()) {
            return redirect()->back()->with('error', 'File tidak valid');
        }

        try {
            $spreadsheet = IOFactory::load($file->getTempName());
            $rows = $spreadsheet->getActiveSheet()->toArray();

            array_shift($rows); // remove header

            $dataInsert = [];
            foreach ($rows as $r) {
                if (empty($r[0])) continue;

                $dataInsert[] = [
                    'nama_lembaga' => $r[0],
                    'created_by'   => session()->get('id'),
                    'updated_by'   => session()->get('id')
                ];
            }

            if ($dataInsert) {
                $this->Lemb_akreditasiModel->insertBatch($dataInsert);
            }

            return redirect()->to('/lemb_akreditasi')
                ->with('success', 'Data berhasil diimpor.');
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Gagal mengimpor: ' . $e->getMessage());
        }
    }

    // -----------------------------
    // TEMPLATE DOWNLOAD
    // -----------------------------
    public function template()
    {
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        $sheet->fromArray(
            ['Nama Lembaga'],   // hanya inputan yang diizinkan
            null,
            'A1'
        );

        $writer = new Xlsx($spreadsheet);
        $filename = "template-lemb_akreditasi.xlsx";

        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header("Content-Disposition: attachment; filename=\"$filename\"");
        $writer->save('php://output');
        exit;
    }
}
