<?php

namespace App\Controllers;

use App\Models\FakultaModel;
use CodeIgniter\Controller;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\IOFactory;
use App\Libraries\LogAktivitas;

class FakultaController extends BaseController
{
    protected $FakultaModel;
    protected $logAktivitas;

    public function __construct()
    {
        $this->FakultaModel = new FakultaModel();
        $this->logAktivitas = new LogAktivitas();
    }

    public function index()
    {
        if (!session()->get('isLoggedIn')) {
            return redirect()->to('/login');
        }

        $permissionCheck = $this->checkPermission('fakultas_view');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data['items'] = $this->FakultaModel->findAll();
        return view('fakultas/index', $this->data + $data);
    }

    public function create()
    {
        $permissionCheck = $this->checkPermission('fakultas_create');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        return view('fakultas/create', $this->data);
    }

    public function store()
    {
        $permissionCheck = $this->checkPermission('fakultas_create');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data = [
            'nama_fakultas' => $this->request->getPost('nama_fakultas'),
            'id_user'       => session()->get('id'),
            'created_by'    => session()->get('id'),
            'updated_by'    => session()->get('id'),
        ];

        // INSERT
        $this->FakultaModel->insert($data);

        // ✅ AMBIL ID ASLI DARI DATABASE
        $idBaru = $this->FakultaModel->getInsertID();

        // REKAP KETERANGAN
        $keterangan = [];
        $keterangan[] = '<strong>Data yang ditambahkan:</strong>';
        $keterangan[] = '<strong>ID</strong>: ' . $idBaru;
        $keterangan[] = '<strong>Nama Fakultas</strong>: ' . esc($data['nama_fakultas']);

        $keteranganLog = implode('<br>', $keterangan);

        // LOG
        $user = $this->getUser();
        $this->logAktivitas->log(
            'create_fakultas',
            $user['id'],
            $user['name'],
            current_url(),
            $keteranganLog
        );

        return redirect()->to('/fakultas')->with('success', 'Data berhasil ditambahkan.');
    }




    public function edit($id)
    {
        $permissionCheck = $this->checkPermission('fakultas_edit');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $item = $this->FakultaModel->find($id);
        $user = $this->getUser();
        $this->logAktivitas->log('edit_fakultas', $user['id'], $user['name'], current_url());
        return view('fakultas/edit', $this->data + ['item' => $item]);
    }

    public function update($id)
    {
        $permissionCheck = $this->checkPermission('fakultas_edit');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        // DATA LAMA
        $old = $this->FakultaModel->find($id);

        $data = [
            'nama_fakultas' => $this->request->getPost('nama_fakultas'),
            'updated_by'    => session()->get('id'),
        ];

        $this->FakultaModel->update($id, $data);

        // REKAP
        $keterangan = [];
        $keterangan[] = '<strong>Data yang diperbarui:</strong>';
        $keterangan[] = '<strong>ID</strong>: ' . $id;

        if ($old['nama_fakultas'] !== $data['nama_fakultas']) {
            $keterangan[] =
                '<strong>Nama Fakultas</strong>:<br>' .
                'Sebelum: "' . esc($old['nama_fakultas']) . '"<br>' .
                'Sesudah: "' . esc($data['nama_fakultas']) . '"';
        }

        $keteranganLog = implode('<hr>', $keterangan);

        $user = $this->getUser();
        $this->logAktivitas->log(
            'update_fakultas',
            $user['id'],
            $user['name'],
            current_url(),
            $keteranganLog
        );

        return redirect()->to('/fakultas')->with('success', 'Data berhasil diperbarui.');
    }


    public function delete($id)
    {
        $permissionCheck = $this->checkPermission('fakultas_delete');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        // DATA SEBELUM DIHAPUS
        $item = $this->FakultaModel->find($id);

        $this->FakultaModel->delete($id);

        // REKAP
        $keterangan = [];
        $keterangan[] = '<strong>Data yang dihapus:</strong>';
        $keterangan[] = '<strong>ID</strong>: ' . $id;
        $keterangan[] = '<strong>Nama Fakultas</strong>: ' . esc($item['nama_fakultas']);

        $keteranganLog = implode('<br>', $keterangan);

        $user = $this->getUser();
        $this->logAktivitas->log(
            'delete_fakultas',
            $user['id'],
            $user['name'],
            current_url(),
            $keteranganLog
        );

        return redirect()->to('/fakultas')->with('success', 'Data berhasil dihapus.');
    }



    public function show($id)
    {
        $permissionCheck = $this->checkPermission('fakultas_view');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $item = $this->FakultaModel->find($id);
        return view('fakultas/show', $this->data);
    }
    private function getUser()
    {
        return [
            'id'   => session()->get('id'),
            'name' => session()->get('name'),
        ];
    }

    // --- FITUR EXCEL ---

    public function export()
    {
        $permissionCheck = $this->checkPermission('fakultas_view'); // Asumsi izin export sama dengan view
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data = $this->FakultaModel->findAll();
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Header Kolom
        $headers = ['id', 'id_fakultas', 'nama_fakultas', 'id_user', 'created_by', 'updated_by'];
        $capitalizedHeaders = array_map(fn($h) => ucwords(str_replace('_', ' ', $h)), $headers);
        $sheet->fromArray($capitalizedHeaders, NULL, 'A1');

        // Data
        $rowNumber = 2;
        foreach ($data as $item) {
            $rowData = [];
            foreach ($headers as $header) {
                $rowData[] = $item[$header] ?? '';
            }
            $sheet->fromArray($rowData, NULL, 'A' . $rowNumber++);
        }

        $writer = new Xlsx($spreadsheet);
        $filename = 'fakultas-export-' . date('Y-m-d') . '.xlsx';

        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header("Content-Disposition: attachment; filename=\"$filename\"");
        header('Cache-Control: max-age=0');
        $writer->save('php://output');
        exit;
    }

    public function import()
    {
        $permissionCheck = $this->checkPermission('fakultas_create'); // Asumsi izin import sama dengan create
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $file = $this->request->getFile('excel_file');
        if ($file && $file->isValid() && !$file->hasMoved()) {
            try {
                $spreadsheet = IOFactory::load($file->getTempName());
                $sheet = $spreadsheet->getActiveSheet();
                $dataRows = $sheet->toArray(null, true, true, true);

                if (count($dataRows) < 2) {
                    session()->setFlashdata('error', 'File Excel kosong atau hanya berisi header.');
                    return redirect()->to(base_url('/fakultas'));
                }

                $headerRow = array_values(array_shift($dataRows));
                $dbFields = array_map(fn($h) => strtolower(str_replace(' ', '_', $h)), $headerRow);

                $importData = [];
                foreach ($dataRows as $row) {
                    if (empty(array_filter(array_values($row)))) continue; // Skip baris kosong
                    $rowData = array_combine($dbFields, array_values($row));
                    $importData[] = $rowData;
                }

                if (!empty($importData)) {
                    $this->FakultaModel->insertBatch($importData);
                    session()->setFlashdata('success', count($importData) . ' baris data berhasil diimpor.');
                } else {
                    session()->setFlashdata('error', 'Tidak ada data valid untuk diimpor.');
                }
            } catch (\Exception $e) {
                session()->setFlashdata('error', 'Gagal memproses file Excel: ' . $e->getMessage());
            }
        } else {
            session()->setFlashdata('error', 'Gagal mengunggah file. Pastikan file valid.');
        }

        return redirect()->to(base_url('/fakultas'));
    }

    public function template()
    {
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Header kolom (hanya allowedFields)
        $headers = ['id_fakultas', 'nama_fakultas', 'id_user', 'created_by', 'updated_by'];
        $capitalizedHeaders = array_map(fn($h) => ucwords(str_replace('_', ' ', $h)), $headers);
        $sheet->fromArray($capitalizedHeaders, NULL, 'A1');

        $writer = new Xlsx($spreadsheet);
        $filename = 'fakultas-template.xlsx';

        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header("Content-Disposition: attachment; filename=\"$filename\"");
        header('Cache-Control: max-age=0');
        $writer->save('php://output');
        exit;
    }
}
