<?php

namespace App\Controllers;

use App\Models\Evaluasi_diriModel;
use App\Models\Standar_lembaga_akreditasiModel;
use App\Models\ProdiModel;
use App\Libraries\LogAktivitas;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\IOFactory;

class Evaluasi_diriController extends BaseController
{
    protected $evaluasiModel;
    protected $standarModel;
    protected $prodiModel;
    protected $logAktivitas;


    public function __construct()
    {
        $this->evaluasiModel = new Evaluasi_diriModel();
        $this->standarModel  = new Standar_lembaga_akreditasiModel();
        $this->prodiModel    = new ProdiModel();
        $this->logAktivitas = new LogAktivitas();
    }

    public function index()
    {
        if (!session()->get('isLoggedIn')) {
            return redirect()->to('/login');
        }

        $permissionCheck = $this->checkPermission('evaluasi_diri_view');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data['items'] = $this->evaluasiModel
            ->select('evaluasi_diri.*, prodi.nama_prodi, standar_lembaga_akreditasi.nama_standar')
            ->join('prodi', 'prodi.id_prodi = evaluasi_diri.id_prodi')
            ->join('standar_lembaga_akreditasi', 'standar_lembaga_akreditasi.id_standar = evaluasi_diri.id_standar')
            ->findAll();

        return view('evaluasi_diri/index', $this->data + $data);
    }

    public function create()
    {
        if (!session()->get('isLoggedIn')) {
            return redirect()->to('/login');
        }

        $permissionCheck = $this->checkPermission('evaluasi_diri_create');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data = [
            'prodi'   => $this->prodiModel->findAll(),
            'standar' => $this->standarModel->findAll()
        ];

        return view('evaluasi_diri/create', $this->data + $data);
    }

    public function store()
    {
        $permissionCheck = $this->checkPermission('evaluasi_diri_create');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data = [
            'id_prodi'          => $this->request->getPost('id_prodi'),
            'id_standar'        => $this->request->getPost('id_standar'),
            'nilai_standar'     => $this->request->getPost('nilai_standar'),
            'faktor_internal'   => $this->request->getPost('faktor_internal'),
            'faktor_eksternal'  => $this->request->getPost('faktor_eksternal'),
            'kesimpulan'        => $this->request->getPost('kesimpulan'),
            'created_by'        => session()->get('id'),        ];

        $this->evaluasiModel->insert($data);

        // ===== LOG REKAP =====
        $fields = [
            'id_prodi'         => 'Prodi',
            'id_standar'       => 'Standar',
            'nilai_standar'    => 'Nilai Standar',
            'faktor_internal'  => 'Faktor Internal',
            'faktor_eksternal' => 'Faktor Eksternal',
            'kesimpulan'       => 'Kesimpulan',
        ];

        $keterangan = [];
        $keterangan[] = '<strong>Data yang ditambahkan:</strong>';

        foreach ($fields as $field => $label) {
            $value = esc($data[$field] ?? '');
            if ($value === '') {
                $value = '<em>(kosong)</em>';
            }
            $keterangan[] = "<strong>{$label}</strong>:<br>\"{$value}\"";
        }

        $user = $this->getUser();
        $this->logAktivitas->log(
            'create_evaluasi_diri',
            $user['id'],
            $user['name'],
            current_url(),
            implode('<hr>', $keterangan)
        );

        return redirect()->to('/evaluasi_diri')
            ->with('success', 'Evaluasi diri berhasil ditambahkan');
    }



    public function edit($id)
    {
        if (!session()->get('isLoggedIn')) {
            return redirect()->to('/login');
        }

        $permissionCheck = $this->checkPermission('evaluasi_diri_edit');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data = [
            'item'   => $this->evaluasiModel->find($id),
            'prodi'  => $this->prodiModel->findAll(),
            'standar' => $this->standarModel->findAll()
        ];

        return view('evaluasi_diri/edit', $this->data + $data);
    }

    public function update($id)
    {
        if (!session()->get('isLoggedIn')) {
            return redirect()->to('/login');
        }

        $permissionCheck = $this->checkPermission('evaluasi_diri_edit');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $before = $this->evaluasiModel->find($id);

        $after = [
            'id_prodi'          => $this->request->getPost('id_prodi'),
            'id_standar'        => $this->request->getPost('id_standar'),
            'nilai_standar'     => $this->request->getPost('nilai_standar'),
            'faktor_internal'   => $this->request->getPost('faktor_internal'),
            'faktor_eksternal'  => $this->request->getPost('faktor_eksternal'),
            'kesimpulan'        => $this->request->getPost('kesimpulan'),
            'updated_by'        => session()->get('id'),
        ];

        $this->evaluasiModel->update($id, $after);

        $fields = [
            'id_prodi'         => 'Prodi',
            'id_standar'       => 'Standar',
            'nilai_standar'    => 'Nilai Standar',
            'faktor_internal'  => 'Faktor Internal',
            'faktor_eksternal' => 'Faktor Eksternal',
            'kesimpulan'       => 'Kesimpulan',
        ];

        $keterangan = [];
        $keterangan[] = '<strong>Perubahan Data Evaluasi Diri:</strong>';

        foreach ($fields as $field => $label) {

            $old = esc($before[$field] ?? '');
            $new = esc($after[$field] ?? '');

            if ($old !== $new) {

                if ($old === '') $old = '<em>(kosong)</em>';
                if ($new === '') $new = '<em>(kosong)</em>';

                if (in_array($field, ['faktor_internal', 'faktor_eksternal', 'kesimpulan'])) {
                    $old = nl2br($old);
                    $new = nl2br($new);
                }

                $keterangan[] =
                    "<strong>{$label}</strong>:<br>" .
                    "Sebelum:<br>\"{$old}\"<br>" .
                    "Sesudah:<br>\"{$new}\"";
            }
        }

        if (count($keterangan) === 1) {
            $keterangan[] = '<em>Tidak ada perubahan data.</em>';
        }

        $user = $this->getUser();
        $this->logAktivitas->log(
            'update_evaluasi_diri',
            $user['id'],
            $user['name'],
            current_url(),
            implode('<hr>', $keterangan)
        );

        return redirect()->to('/evaluasi_diri')
            ->with('success', 'Evaluasi diri berhasil diperbarui.');
    }


    public function delete($id)
    {
        if (!session()->get('isLoggedIn')) {
            return redirect()->to('/login');
        }

        $permissionCheck = $this->checkPermission('evaluasi_diri_delete');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $item = $this->evaluasiModel->find($id);

        if (!$item) {
            return redirect()->to('/evaluasi_diri')
                ->with('error', 'Data tidak ditemukan.');
        }

        $this->evaluasiModel->delete($id);

        $fields = [
            'id_prodi'         => 'Prodi',
            'id_standar'       => 'Standar',
            'nilai_standar'    => 'Nilai Standar',
            'faktor_internal'  => 'Faktor Internal',
            'faktor_eksternal' => 'Faktor Eksternal',
            'kesimpulan'       => 'Kesimpulan',
        ];

        $keterangan = [];
        $keterangan[] = '<strong>Data Evaluasi Diri yang dihapus:</strong>';

        foreach ($fields as $field => $label) {
            $value = esc($item[$field] ?? '');
            if ($value === '') {
                $value = '<em>(kosong)</em>';
            }

            if (in_array($field, ['faktor_internal', 'faktor_eksternal', 'kesimpulan'])) {
                $value = nl2br($value);
            }

            $keterangan[] =
                "<strong>{$label}</strong>:<br>" .
                "\"{$value}\"";
        }

        $user = $this->getUser();
        $this->logAktivitas->log(
            'delete_evaluasi_diri',
            $user['id'],
            $user['name'],
            current_url(),
            implode('<hr>', $keterangan)
        );

        return redirect()->to('/evaluasi_diri')
            ->with('success', 'Data berhasil dihapus.');
    }

    public function show($id)
    {
        $permissionCheck = $this->checkPermission('evaluasi_diri_view');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data['item'] = $this->evaluasiModel->find($id);
        return view('evaluasi_diri/show', $data);
    }

    private function getUser()
    {
        return [
            'id'   => session()->get('id'),
            'name' => session()->get('name'),
        ];
    }


    // --- FITUR EXCEL ---

    public function export()
    {
        $permissionCheck = $this->checkPermission('evaluasi_diri_view'); // Asumsi izin export sama dengan view
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data = $this->evaluasiModel->findAll();
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Header Kolom
        $headers = ['id', 'id_evaluasi', 'id_standar', 'id_prodi', 'nilai_standar', 'faktor_internal', 'faktor_eksternal', 'kesimpulan', 'created_by', 'updated_by'];
        $capitalizedHeaders = array_map(fn($h) => ucwords(str_replace('_', ' ', $h)), $headers);
        $sheet->fromArray($capitalizedHeaders, NULL, 'A1');

        // Data
        $rowNumber = 2;
        foreach ($data as $item) {
            $rowData = [];
            foreach ($headers as $header) {
                $rowData[] = $item[$header] ?? '';
            }
            $sheet->fromArray($rowData, NULL, 'A' . $rowNumber++);
        }

        $writer = new Xlsx($spreadsheet);
        $filename = 'evaluasi_diri-export-' . date('Y-m-d') . '.xlsx';

        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header("Content-Disposition: attachment; filename=\"$filename\"");
        header('Cache-Control: max-age=0');
        $writer->save('php://output');
        exit;
    }

    public function import()
    {
        $permissionCheck = $this->checkPermission('evaluasi_diri_create'); // Asumsi izin import sama dengan create
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $file = $this->request->getFile('excel_file');
        if ($file && $file->isValid() && !$file->hasMoved()) {
            try {
                $spreadsheet = IOFactory::load($file->getTempName());
                $sheet = $spreadsheet->getActiveSheet();
                $dataRows = $sheet->toArray(null, true, true, true);

                if (count($dataRows) < 2) {
                    session()->setFlashdata('error', 'File Excel kosong atau hanya berisi header.');
                    return redirect()->to(base_url('/evaluasi_diri'));
                }

                $headerRow = array_values(array_shift($dataRows));
                $dbFields = array_map(fn($h) => strtolower(str_replace(' ', '_', $h)), $headerRow);

                $importData = [];
                foreach ($dataRows as $row) {
                    if (empty(array_filter(array_values($row)))) continue; // Skip baris kosong
                    $rowData = array_combine($dbFields, array_values($row));
                    $importData[] = $rowData;
                }

                if (!empty($importData)) {
                    $this->evaluasiModel->insertBatch($importData);
                    session()->setFlashdata('success', count($importData) . ' baris data berhasil diimpor.');
                } else {
                    session()->setFlashdata('error', 'Tidak ada data valid untuk diimpor.');
                }
            } catch (\Exception $e) {
                session()->setFlashdata('error', 'Gagal memproses file Excel: ' . $e->getMessage());
            }
        } else {
            session()->setFlashdata('error', 'Gagal mengunggah file. Pastikan file valid.');
        }

        return redirect()->to(base_url('/evaluasi_diri'));
    }

    public function template()
    {
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Header kolom (hanya allowedFields)
        $headers = ['id_evaluasi', 'id_standar', 'id_prodi', 'nilai_standar', 'faktor_internal', 'faktor_eksternal', 'kesimpulan', 'created_by', 'updated_by'];
        $capitalizedHeaders = array_map(fn($h) => ucwords(str_replace('_', ' ', $h)), $headers);
        $sheet->fromArray($capitalizedHeaders, NULL, 'A1');

        $writer = new Xlsx($spreadsheet);
        $filename = 'evaluasi_diri-template.xlsx';

        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header("Content-Disposition: attachment; filename=\"$filename\"");
        header('Cache-Control: max-age=0');
        $writer->save('php://output');
        exit;
    }
}
