<?php

namespace App\Controllers;

use App\Controllers\BaseController;
use App\Models\EmailConfigModel;
use App\Libraries\LogAktivitas;


class EmailConfigController extends BaseController
{
    protected $emailConfigModel;
    protected $logAktivitas;

    public function __construct()
    {
        helper(['form', 'url']);
        $this->emailConfigModel = new EmailConfigModel();
        $this->logAktivitas = new LogAktivitas();
    }

    public function index()
    {
        $config = $this->emailConfigModel->getActiveConfig();

        if (!$config) {
            // Create default config if none exists
            $defaultConfig = [
                'protocol' => 'smtp',
                'smtp_host' => 'smtp.gmail.com',
                'smtp_user' => '',
                'smtp_pass' => '',
                'smtp_port' => 587,
                'smtp_crypto' => 'tls',
                'from_email' => '',
                'from_name' => 'AUDIT TERPADU',
                'is_active' => 1,
            ];
            $this->emailConfigModel->insert($defaultConfig);
            $config = $this->emailConfigModel->getActiveConfig();
        }
        $user = $this->getUser();
        $this->logAktivitas->log('akses_email', $user['id'], $user['name'], current_url());

        return view('email_config/index', $this->data + [
            'title' => 'Email Configuration',
            'config' => $config
        ]);
    }

    public function edit($id)
    {
        $config = $this->emailConfigModel->find($id);

        if (!$config) {
            return redirect()->to('/email-config')->with('error', 'Configuration not found');
        }
        $user = $this->getUser();
        $this->logAktivitas->log('edit_email', $user['id'], $user['name'], current_url());

        return view('email_config/edit', $this->data + [
            'title' => 'Edit Email Configuration',
            'config' => $config
        ]);
    }

    public function update($id)
    {
        $rules = [
            'protocol' => 'required',
            'smtp_host' => 'required',
            'smtp_user' => 'permit_empty',
            'smtp_pass' => 'permit_empty',
            'smtp_port' => 'required|numeric',
            'smtp_crypto' => 'permit_empty',
            'from_email' => 'required|valid_email',
            'from_name' => 'required',
        ];

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('errors', $this->validator->getErrors());
        }

        $data = [
            'protocol' => $this->request->getPost('protocol'),
            'smtp_host' => $this->request->getPost('smtp_host'),
            'smtp_user' => $this->request->getPost('smtp_user'),
            'smtp_pass' => $this->request->getPost('smtp_pass'),
            'smtp_port' => $this->request->getPost('smtp_port'),
            'smtp_crypto' => $this->request->getPost('smtp_crypto'),
            'from_email' => $this->request->getPost('from_email'),
            'from_name' => $this->request->getPost('from_name'),
        ];

        if ($this->emailConfigModel->update($id, $data)) {
            return redirect()->to('/email-config')->with('success', 'Email configuration updated successfully');
        } else {
            return redirect()->back()->withInput()->with('error', 'Failed to update configuration');
        }
        
    }
    private function getUser()
    {
        return [
            'id'   => session()->get('id'),
            'name' => session()->get('name'),
        ];
    }
    // Test email sending
    public function testEmail()
    {
        $config = $this->emailConfigModel->getActiveConfig();

        if (!$config) {
            return $this->response->setJSON(['success' => false, 'message' => 'No active email configuration found']);
        }

        // Configure email service
        $emailService = \Config\Services::email();
        $emailService->initialize([
            'protocol' => $config['protocol'],
            'SMTPHost' => $config['smtp_host'],
            'SMTPUser' => $config['smtp_user'],
            'SMTPPass' => $config['smtp_pass'],
            'SMTPPort' => $config['smtp_port'],
            'SMTPCrypto' => $config['smtp_crypto'],
            'mailType' => 'html',
            'charset' => 'utf-8',
        ]);

        $emailService->setFrom($config['from_email'], $config['from_name']);
        $emailService->setTo($config['from_email']); // Send to self for testing
        $emailService->setSubject('Email Configuration Test');
        $emailService->setMessage('<h3>Email Configuration Test</h3><p>This is a test email to verify your email configuration is working correctly.</p>');

        if ($emailService->send()) {
            return $this->response->setJSON(['success' => true, 'message' => 'Test email sent successfully']);
        } else {
            return $this->response->setJSON(['success' => false, 'message' => 'Failed to send test email: ' . $emailService->printDebugger()]);
        }
    }
}
