<?php

namespace App\Controllers;

use App\Models\DosenModel;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\IOFactory;
use App\Libraries\LogAktivitas;


class DosenController extends BaseController
{
    protected $model;
    protected $logAktivitas;

    public function __construct()
    {
        $this->model = new DosenModel();
        $this->logAktivitas = new LogAktivitas();
    }

    public function index()
    {
        $this->checkPermission('dosen_view');

        $data['items'] = $this->model->findAll();

        return view('dosen/index', $this->data + $data);
    }


    public function create()
    {
        $permissionCheck = $this->checkPermission('dosen_create');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        return view('dosen/create', $this->data);
    }

    public function store()
    {
        $permissionCheck = $this->checkPermission('dosen_create');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }
        $post = $this->request->getPost();

        $post['user_id']    = session()->get('id');
        $post['created_by'] = session()->get('id');

        $fields = [
            'nama_lengkap' => 'Nama',
            'nidn'         => 'NIDN',
            'niy'          => 'NIY',
            'alamat'       => 'Alamat',
        ];

        foreach ($fields as $field => $label) {
            if (!empty($post[$field])) {
                $value = esc($post[$field]);

                if ($field === 'alamat') {
                    $value = nl2br($value);
                }

                $keterangan[] =
                    "<strong>{$label}</strong>:<br>" .
                    "\"{$value}\"";
            }
        }

        $keteranganLog = implode('<hr>', $keterangan);
        $this->model->insert($post);
        $user = $this->getUser();
        $this->logAktivitas->log(
            'create_dosen',
            $user['id'],
            $user['name'],
            current_url(),
            $keteranganLog
        );

        return redirect()->to('/dosen')->with('success', 'Data berhasil ditambahkan.');
    }



    public function edit($id)
    {
        $this->checkPermission('dosen_edit');

        $item = $this->model->find($id);
        if (!$item) {
            return redirect()->to('/dosen')->with('error', 'Data tidak ditemukan');
        }

        return view('dosen/edit', $this->data + ['item' => $item]);
    }

    public function update($id)
    {
        $permissionCheck = $this->checkPermission('dosen_edit');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $lama = $this->model->find($id);
        if (!$lama) {
            return redirect()->to('/dosen')->with('error', 'Data tidak ditemukan');
        }

        $post = $this->request->getPost();

        $perubahan = [];

        $fields = [
            'nama_lengkap' => 'Nama',
            'nidn'         => 'NIDN',
            'niy'          => 'NIY',
            'alamat'       => 'Alamat',
        ];

        foreach ($fields as $field => $label) {
            if (
                isset($post[$field]) &&
                isset($lama[$field]) &&
                $lama[$field] !== $post[$field]
            ) {
                $perubahan[] =
                    "<strong>{$label}</strong>:<br>" .
                    "Lama: \"{$lama[$field]}\"<br>" .
                    "Baru: \"{$post[$field]}\"";
            }
        }

        $keterangan = empty($perubahan)
            ? 'Menyimpan data tanpa perubahan nilai'
            : implode('<hr>', $perubahan);

        $post['updated_by'] = session()->get('id');
        $this->model->update($id, $post);
        $user = $this->getUser();
        $this->logAktivitas->log(
            'update_dosen',
            $user['id'],
            $user['name'],
            current_url(),
            $keterangan
        );

        return redirect()->to('/dosen')->with('success', 'Data berhasil diperbarui.');
    }


    public function delete($id)
    {
        $this->checkPermission('dosen_delete');

        $item = $this->model->find($id);
        $this->model->delete($id);

        $user = $this->getUser();
        $this->logAktivitas->log(
            'delete_dosen',
            $user['id'],
            $user['name'],
            current_url(),
            'Menghapus dosen: ' . $item['nama_lengkap'] .
                ' (NIDN: ' . $item['nidn'] . ')'
        );

        return redirect()->to('/dosen')->with('success', 'Data berhasil dihapus.');
    }
    private function getUser()
    {
        return [
            'id'   => session()->get('id'),
            'name' => session()->get('name')
        ];
    }

    public function show($id)
    {
        $permissionCheck = $this->checkPermission('dosen_view');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $item = $this->model->find($id);
        if (!$item) {
            return redirect()->to('/dosen')->with('error', 'Data tidak ditemukan');
        }
        return view('dosen/show', $this->data + ['item' => $item]);
    }

    // --- FITUR EXCEL ---

    public function export()
    {
        $permissionCheck = $this->checkPermission('dosen_view'); // Asumsi izin export sama dengan view
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data = $this->model->findAll();
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Header Kolom
        $headers = ['id', 'id_dosen', 'nidn', 'niy', 'nama_lengkap', 'alamat', 'user_id', 'created_by', 'updated_by'];
        $capitalizedHeaders = array_map(fn($h) => ucwords(str_replace('_', ' ', $h)), $headers);
        $sheet->fromArray($capitalizedHeaders, NULL, 'A1');

        // Data
        $rowNumber = 2;
        foreach ($data as $item) {
            $rowData = [];
            foreach ($headers as $header) {
                $rowData[] = $item[$header] ?? '';
            }
            $sheet->fromArray($rowData, NULL, 'A' . $rowNumber++);
        }

        $writer = new Xlsx($spreadsheet);
        $filename = 'dosen-export-' . date('Y-m-d') . '.xlsx';

        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header("Content-Disposition: attachment; filename=\"$filename\"");
        header('Cache-Control: max-age=0');
        $writer->save('php://output');
        exit;
    }

    public function import()
    {
        $permissionCheck = $this->checkPermission('dosen_create'); // Asumsi izin import sama dengan create
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $file = $this->request->getFile('excel_file');
        if ($file && $file->isValid() && !$file->hasMoved()) {
            try {
                $spreadsheet = IOFactory::load($file->getTempName());
                $sheet = $spreadsheet->getActiveSheet();
                $dataRows = $sheet->toArray(null, true, true, true);

                if (count($dataRows) < 2) {
                    session()->setFlashdata('error', 'File Excel kosong atau hanya berisi header.');
                    return redirect()->to(base_url('/dosen'));
                }

                $headerRow = array_values(array_shift($dataRows));
                $dbFields = array_map(fn($h) => strtolower(str_replace(' ', '_', $h)), $headerRow);

                $importData = [];
                foreach ($dataRows as $row) {
                    if (empty(array_filter(array_values($row)))) continue; // Skip baris kosong
                    $rowData = array_combine($dbFields, array_values($row));
                    $importData[] = $rowData;
                }

                if (!empty($importData)) {
                    $this->model->insertBatch($importData);
                    session()->setFlashdata('success', count($importData) . ' baris data berhasil diimpor.');
                } else {
                    session()->setFlashdata('error', 'Tidak ada data valid untuk diimpor.');
                }
            } catch (\Exception $e) {
                session()->setFlashdata('error', 'Gagal memproses file Excel: ' . $e->getMessage());
            }
        } else {
            session()->setFlashdata('error', 'Gagal mengunggah file. Pastikan file valid.');
        }

        return redirect()->to(base_url('/dosen'));
    }

    public function template()
    {
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Header kolom (hanya allowedFields)
        $headers = ['id_dosen', 'nidn', 'niy', 'nama_lengkap', 'alamat', 'user_id', 'created_by', 'updated_by'];
        $capitalizedHeaders = array_map(fn($h) => ucwords(str_replace('_', ' ', $h)), $headers);
        $sheet->fromArray($capitalizedHeaders, NULL, 'A1');

        $writer = new Xlsx($spreadsheet);
        $filename = 'dosen-template.xlsx';

        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header("Content-Disposition: attachment; filename=\"$filename\"");
        header('Cache-Control: max-age=0');
        $writer->save('php://output');
        exit;
    }
}
