<?php

namespace App\Controllers;

use App\Models\Daftar_tilikModel;
use App\Models\Standar_lembaga_akreditasiModel;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\IOFactory;
use App\Libraries\LogAktivitas;


class Daftar_tilikController extends BaseController
{
    protected $Daftar_tilikModel;
    protected $StandarModel;
    protected $logAktivitas;


    public function __construct()
    {
        $this->Daftar_tilikModel = new Daftar_tilikModel();
        $this->StandarModel     = new Standar_lembaga_akreditasiModel();
        $this->logAktivitas = new LogAktivitas();
    }

    public function index()
    {
        if (!session()->get('isLoggedIn')) {
            return redirect()->to('/login');
        }

        $permissionCheck = $this->checkPermission('daftar_tilik_view');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data['items'] = $this->Daftar_tilikModel->findAll();
        return view('daftar_tilik/index', $this->data + $data);
    }

    public function create()
    {
        $permissionCheck = $this->checkPermission('daftar_tilik_create');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        // untuk dropdown standar
        $data['standar'] = $this->StandarModel->findAll();
        return view('daftar_tilik/create', $this->data + $data);
    }

    public function store()
    {
        // cek permission
        $permissionCheck = $this->checkPermission('daftar_tilik_create');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        // data dari form
        $data = [
            'id_standar'      => $this->request->getPost('id_standar'),
            'pertanyaan'      => $this->request->getPost('pertanyaan'),
            'jawaban'         => $this->request->getPost('jawaban'),
            'link_bukti'      => $this->request->getPost('link_bukti'),
            'hasil_observasi' => $this->request->getPost('hasil_observasi'),
            'hasil_temuan'    => $this->request->getPost('hasil_temuan'),
            'created_by'      => session()->get('id'),
            'updated_by'      => session()->get('id'),
        ];

        // =======================
        // MAPPING HASIL TEMUAN
        // =======================
        $mapHasilTemuan = [
            'sesuai'              => 'Sesuai',
            'tidak_sesuai'        => 'Tidak Sesuai',
            'tidak_sesuai_mayor'  => 'Tidak Sesuai Mayor',
        ];

        // =======================
        // REKAP LOG CREATE
        // =======================
        $fields = [
            'id_standar'      => 'Standar',
            'pertanyaan'      => 'Pertanyaan',
            'jawaban'         => 'Jawaban',
            'link_bukti'      => 'Link Bukti',
            'hasil_observasi' => 'Hasil Observasi',
            'hasil_temuan'    => 'Hasil Temuan',
        ];

        $keterangan = [];

        foreach ($fields as $field => $label) {
            if (isset($data[$field])) {

                // khusus hasil_temuan pakai label
                if ($field === 'hasil_temuan') {
                    $value = $mapHasilTemuan[$data[$field]] ?? '<em>(belum dipilih)</em>';

                    // tambahkan <br> supaya konsisten
                    $value = nl2br($value);
                } else {
                    $value = esc($data[$field]);

                    if (in_array($field, ['pertanyaan', 'jawaban', 'hasil_observasi'])) {
                        $value = nl2br($value);
                    }

                    if ($value === '') {
                        $value = '<em>(kosong)</em>';
                    }
                }

                $keterangan[] =
                    "<strong>{$label}</strong>:<br>" .
                    "\"{$value}\"";
            }
        }

        $keteranganLog = implode('<hr>', $keterangan);

        // simpan data
        $this->Daftar_tilikModel->insert($data);

        // simpan log
        $user = $this->getUser();
        $this->logAktivitas->log(
            'create_daftar_tilik',
            $user['id'],
            $user['name'],
            current_url(),
            $keteranganLog
        );

        return redirect()->to('/daftar_tilik')
            ->with('success', 'Data berhasil ditambahkan.');
    }


    public function edit($id)
    {
        $permissionCheck = $this->checkPermission('daftar_tilik_edit');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data['item']    = $this->Daftar_tilikModel->find($id);
        $data['standar'] = $this->StandarModel->findAll();

        return view('daftar_tilik/edit', $this->data + $data);
    }

    public function update($id)
    {
        // cek permission
        $permissionCheck = $this->checkPermission('daftar_tilik_edit');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        // ambil data lama
        $oldData = $this->Daftar_tilikModel->find($id);

        if (!$oldData) {
            return redirect()->to('/daftar_tilik')
                ->with('error', 'Data tidak ditemukan.');
        }

        // data baru
        $newData = [
            'id_standar'      => $this->request->getPost('id_standar'),
            'pertanyaan'      => $this->request->getPost('pertanyaan'),
            'jawaban'         => $this->request->getPost('jawaban'),
            'link_bukti'      => $this->request->getPost('link_bukti'),
            'hasil_observasi' => $this->request->getPost('hasil_observasi'),
            'hasil_temuan'    => $this->request->getPost('hasil_temuan'),
            'updated_by'      => session()->get('id'),
        ];

        // =======================
        // MAPPING HASIL TEMUAN
        // =======================
        $mapHasilTemuan = [
            'sesuai'             => 'Sesuai',
            'tidak_sesuai'       => 'Tidak Sesuai',
            'tidak_sesuai_mayor' => 'Tidak Sesuai Mayor',
        ];

        // =======================
        // REKAP PERUBAHAN LOG
        // =======================
        $fields = [
            'id_standar'      => 'Standar',
            'pertanyaan'      => 'Pertanyaan',
            'jawaban'         => 'Jawaban',
            'link_bukti'      => 'Link Bukti',
            'hasil_observasi' => 'Hasil Observasi',
            'hasil_temuan'    => 'Hasil Temuan',
        ];

        $keterangan = [];

        foreach ($fields as $field => $label) {
            $oldValue = $oldData[$field] ?? '';
            $newValue = $newData[$field] ?? '';

            if ($oldValue != $newValue) {

                // khusus hasil_temuan pakai label
                if ($field === 'hasil_temuan') {
                    $oldText = $mapHasilTemuan[$oldValue] ?? '<em>(kosong)</em>';
                    $newText = $mapHasilTemuan[$newValue] ?? '<em>(kosong)</em>';
                } else {
                    $oldText = esc($oldValue);
                    $newText = esc($newValue);

                    if (in_array($field, ['pertanyaan', 'jawaban', 'hasil_observasi'])) {
                        $oldText = nl2br($oldText);
                        $newText = nl2br($newText);
                    }

                    if ($oldText === '') $oldText = '<em>(kosong)</em>';
                    if ($newText === '') $newText = '<em>(kosong)</em>';
                }

                $keterangan[] =
                    "<strong>{$label}</strong>:<br>" .
                    "<em>Sebelum</em>: \"{$oldText}\"<br>" .
                    "<em>Sesudah</em>: \"{$newText}\"";
            }
        }

        $keteranganLog = !empty($keterangan)
            ? implode('<hr>', $keterangan)
            : 'Tidak ada perubahan data.';

        // update data
        $this->Daftar_tilikModel->update($id, $newData);

        // simpan log
        $user = $this->getUser();
        $this->logAktivitas->log(
            'update_daftar_tilik',
            $user['id'],
            $user['name'],
            current_url(),
            $keteranganLog
        );

        return redirect()->to('/daftar_tilik')
            ->with('success', 'Data berhasil diperbarui.');
    }


    public function delete($id)
    {
        // cek permission
        $permissionCheck = $this->checkPermission('daftar_tilik_delete');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        // ambil data sebelum dihapus
        $data = $this->Daftar_tilikModel->find($id);

        if (!$data) {
            return redirect()->to('/daftar_tilik')
                ->with('error', 'Data tidak ditemukan.');
        }

        // =======================
        // MAPPING HASIL TEMUAN
        // =======================
        $mapHasilTemuan = [
            'sesuai'             => 'Sesuai',
            'tidak_sesuai'       => 'Tidak Sesuai',
            'tidak_sesuai_mayor' => 'Tidak Sesuai Mayor',
        ];

        // =======================
        // REKAP LOG DELETE
        // =======================
        $fields = [
            'id_standar'      => 'Standar',
            'pertanyaan'      => 'Pertanyaan',
            'jawaban'         => 'Jawaban',
            'link_bukti'      => 'Link Bukti',
            'hasil_observasi' => 'Hasil Observasi',
            'hasil_temuan'    => 'Hasil Temuan',
        ];

        $keterangan = [];

        // judul log
        $keterangan[] = '<strong>Data yang dihapus:</strong>';

        foreach ($fields as $field => $label) {
            if (isset($data[$field])) {

                if ($field === 'hasil_temuan') {
                    $value = $mapHasilTemuan[$data[$field]] ?? '<em>(kosong)</em>';
                } else {
                    $value = esc($data[$field]);

                    if (in_array($field, ['pertanyaan', 'jawaban', 'hasil_observasi'])) {
                        $value = nl2br($value);
                    }

                    if ($value === '') {
                        $value = '<em>(kosong)</em>';
                    }
                }

                $keterangan[] =
                    "<strong>{$label}</strong>:<br>" .
                    "\"{$value}\"";
            }
        }

        $keteranganLog = implode('<hr>', $keterangan);

        // hapus data
        $this->Daftar_tilikModel->delete($id);

        // simpan log
        $user = $this->getUser();
        $this->logAktivitas->log(
            'delete_daftar_tilik',
            $user['id'],
            $user['name'],
            current_url(),
            $keteranganLog
        );

        return redirect()->to('/daftar_tilik')
            ->with('success', 'Data berhasil dihapus.');
    }


    public function show($id)
    {
        $permissionCheck = $this->checkPermission('daftar_tilik_view');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data['item'] = $this->Daftar_tilikModel->find($id);
        return view('daftar_tilik/show', $this->data + $data);
    }

    private function getUser()
    {
        return [
            'id'   => session()->get('id'),
            'name' => session()->get('name')
        ];
    }
    // --- FITUR EXCEL ---

    public function export()
    {
        $permissionCheck = $this->checkPermission('daftar_tilik_view'); // Asumsi izin export sama dengan view
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data = $this->Daftar_tilikModel->findAll();
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Header Kolom
        $headers = ['id', 'id_tilik', 'id_standar', 'pertanyaan', 'jawaban', 'link_bukti', 'hasil_observasi', 'hasil_temuan', 'created_by', 'updated_by'];
        $capitalizedHeaders = array_map(fn($h) => ucwords(str_replace('_', ' ', $h)), $headers);
        $sheet->fromArray($capitalizedHeaders, NULL, 'A1');

        // Data
        $rowNumber = 2;
        foreach ($data as $item) {
            $rowData = [];
            foreach ($headers as $header) {
                $rowData[] = $item[$header] ?? '';
            }
            $sheet->fromArray($rowData, NULL, 'A' . $rowNumber++);
        }

        $writer = new Xlsx($spreadsheet);
        $filename = 'daftar_tilik-export-' . date('Y-m-d') . '.xlsx';

        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header("Content-Disposition: attachment; filename=\"$filename\"");
        header('Cache-Control: max-age=0');
        $writer->save('php://output');
        exit;
    }

    public function import()
    {
        $permissionCheck = $this->checkPermission('daftar_tilik_create'); // Asumsi izin import sama dengan create
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $file = $this->request->getFile('excel_file');
        if ($file && $file->isValid() && !$file->hasMoved()) {
            try {
                $spreadsheet = IOFactory::load($file->getTempName());
                $sheet = $spreadsheet->getActiveSheet();
                $dataRows = $sheet->toArray(null, true, true, true);

                if (count($dataRows) < 2) {
                    session()->setFlashdata('error', 'File Excel kosong atau hanya berisi header.');
                    return redirect()->to(base_url('/daftar_tilik'));
                }

                $headerRow = array_values(array_shift($dataRows));
                $dbFields = array_map(fn($h) => strtolower(str_replace(' ', '_', $h)), $headerRow);

                $importData = [];
                foreach ($dataRows as $row) {
                    if (empty(array_filter(array_values($row)))) continue; // Skip baris kosong
                    $rowData = array_combine($dbFields, array_values($row));
                    $importData[] = $rowData;
                }

                if (!empty($importData)) {
                    $this->Daftar_tilikModel->insertBatch($importData);
                    session()->setFlashdata('success', count($importData) . ' baris data berhasil diimpor.');
                } else {
                    session()->setFlashdata('error', 'Tidak ada data valid untuk diimpor.');
                }
            } catch (\Exception $e) {
                session()->setFlashdata('error', 'Gagal memproses file Excel: ' . $e->getMessage());
            }
        } else {
            session()->setFlashdata('error', 'Gagal mengunggah file. Pastikan file valid.');
        }

        return redirect()->to(base_url('/daftar_tilik'));
    }

    public function template()
    {
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Header kolom (hanya allowedFields)
        $headers = ['id_tilik', 'id_standar', 'pertanyaan', 'jawaban', 'link_bukti', 'hasil_observasi', 'hasil_temuan', 'created_by', 'updated_by'];
        $capitalizedHeaders = array_map(fn($h) => ucwords(str_replace('_', ' ', $h)), $headers);
        $sheet->fromArray($capitalizedHeaders, NULL, 'A1');

        $writer = new Xlsx($spreadsheet);
        $filename = 'daftar_tilik-template.xlsx';

        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header("Content-Disposition: attachment; filename=\"$filename\"");
        header('Cache-Control: max-age=0');
        $writer->save('php://output');
        exit;
    }
}
