<?php

namespace App\Controllers;

use CodeIgniter\Controller;
use Config\Database;
use PhpOffice\PhpSpreadsheet\IOFactory;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;

class CodeGeneratorController extends BaseController
{
    protected $db;
    protected $forge;

    public function __construct()
    {
        parent::initController(
            service('request'),
            service('response'),
            service('logger')
        );
        $this->db = Database::connect();
        $this->forge = \Config\Database::forge();
    }

    public function index()
    {
        $permissionCheck = $this->checkPermission('codegen_view');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $session = session();
        if (!$session->get('isLoggedIn')) {
            return redirect()->to('/login');
        }
        $tables = $this->db->listTables();

        $data = $this->data;
        $data['tables'] = $tables;

        echo view('codegenerator/index', $data);
    }

    public function generate()
    {
        $permissionCheck = $this->checkPermission('codegen_create');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        // Log activity
        log_message('info', 'CodeGenerator: User ' . session()->get('name') . ' started generating CRUD for table: ' . $this->request->getPost('table'));

        $table = $this->request->getPost('table');
        if (empty($table)) {
            session()->setFlashdata('error', 'Nama tabel harus dipilih.');
            return redirect()->to('/codegenerator');
        }

        // Cek apakah tabel ada, jika tidak buat migrasi
        if (!$this->db->tableExists($table)) {
            $this->createMigration($table);
            try {
                \Config\Services::migrations()->latest();
            } catch (\Exception $e) {
                session()->setFlashdata('error', 'Gagal migrasi: ' . $e->getMessage());
                return redirect()->to('/codegenerator');
            }
        }

        // Ambil ulang fields
        $fields = $this->db->getFieldData($table);
        $primaryKey = 'id';

        $singular = ucfirst(rtrim($table, 's'));
        $modelName = $singular . 'Model';
        $controllerName = $singular . 'Controller';
        $viewFolder = strtolower($table);

        // Output folder
        $basePath = APPPATH . 'Output_CRUD/';
        $modelDir = $basePath . 'Models/';
        $controllerDir = $basePath . 'Controllers/';
        $viewDir = $basePath . 'Views/' . $viewFolder;

        // Pastikan folder tersedia
        if (!is_dir($modelDir)) mkdir($modelDir, 0777, true);
        if (!is_dir($controllerDir)) mkdir($controllerDir, 0777, true);
        if (!is_dir($viewDir)) mkdir($viewDir, 0777, true);

        // === Generate allowedFields (kolom yang bisa diisi selain primary key & timestamps) ===
        $allowedFields = array_filter(
            array_map(fn($f) => $f->name, $fields),
            fn($name) => !in_array($name, ['id', 'created_at', 'updated_at', 'deleted_at'])
        );
        $allowedFieldsString = "'" . implode("','", $allowedFields) . "'";

        // === Definisikan header untuk ekspor dan template ===
        $exportHeaders = array_merge(['id'], $allowedFields);
        $exportHeadersString = "['" . implode("', '", $exportHeaders) . "']";
        $templateHeadersString = "['" . implode("', '", $allowedFields) . "']";


        // === Generate Model ===
        $modelCode = <<<PHP
<?php namespace App\Models;

use CodeIgniter\Model;

class $modelName extends Model
{
    protected \$table = '$table';
    protected \$primaryKey = '$primaryKey';
    protected \$allowedFields = [$allowedFieldsString];
    protected \$useTimestamps = true;
    protected \$returnType = 'array';
}
PHP;

        file_put_contents($modelDir . $modelName . '.php', $modelCode);


        // === Generate Controller ===
        $controllerCode = <<<PHP
<?php namespace App\Controllers;

use App\Models\\$modelName;
use CodeIgniter\Controller;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\IOFactory;

class $controllerName extends BaseController
{
    protected \${$modelName};

    public function __construct()
    {
        \$this->model = new $modelName();
    }
    
    public function index()
    {
        \$session = session();
        if (!\$session->get('isLoggedIn')) {
            return redirect()->to('/login');
        }

        \$permissionCheck = \$this->checkPermission('{$viewFolder}_view');
        if (\$permissionCheck instanceof \\CodeIgniter\\HTTP\\RedirectResponse) {
            return \$permissionCheck;
        }
        
        \$data['items'] = \$this->model->findAll();
        return view('$viewFolder/index', \$data);
    }

    public function create()
    {
        \$permissionCheck = \$this->checkPermission('{$viewFolder}_create');
        if (\$permissionCheck instanceof \\CodeIgniter\\HTTP\\RedirectResponse) {
            return \$permissionCheck;
        }

        return view('$viewFolder/create');
    }

    public function store()
    {
        \$permissionCheck = \$this->checkPermission('{$viewFolder}_create');
        if (\$permissionCheck instanceof \\CodeIgniter\\HTTP\\RedirectResponse) {
            return \$permissionCheck;
        }
        
        \$this->model->insert(\$this->request->getPost());
        return redirect()->to('/$viewFolder')->with('success', 'Data berhasil ditambahkan.');
    }

    public function edit(\$id)
    {
        \$permissionCheck = \$this->checkPermission('{$viewFolder}_edit');
        if (\$permissionCheck instanceof \\CodeIgniter\\HTTP\\RedirectResponse) {
            return \$permissionCheck;
        }

        \$data['item'] = \$this->model->find(\$id);
        return view('$viewFolder/edit', \$data);
    }

    public function update(\$id)
    {
        \$permissionCheck = \$this->checkPermission('{$viewFolder}_edit');
        if (\$permissionCheck instanceof \\CodeIgniter\\HTTP\\RedirectResponse) {
            return \$permissionCheck;
        }

        \$this->model->update(\$id, \$this->request->getPost());
        return redirect()->to('/$viewFolder')->with('success', 'Data berhasil diperbarui.');
    }

    public function delete(\$id)
    {
        \$permissionCheck = \$this->checkPermission('{$viewFolder}_delete');
        if (\$permissionCheck instanceof \\CodeIgniter\\HTTP\\RedirectResponse) {
            return \$permissionCheck;
        }

        \$this->model->delete(\$id);
        return redirect()->to('/$viewFolder')->with('success', 'Data berhasil dihapus.');
    }

    public function show(\$id)
    {
        \$permissionCheck = \$this->checkPermission('{$viewFolder}_view');
        if (\$permissionCheck instanceof \\CodeIgniter\\HTTP\\RedirectResponse) {
            return \$permissionCheck;
        }

        \$data['item'] = \$this->model->find(\$id);
        return view('$viewFolder/show', \$data);
    }
    
    // --- FITUR EXCEL ---

    public function export()
    {
        \$permissionCheck = \$this->checkPermission('{$viewFolder}_view'); // Asumsi izin export sama dengan view
        if (\$permissionCheck instanceof \\CodeIgniter\\HTTP\\RedirectResponse) {
            return \$permissionCheck;
        }

        \$data = \$this->model->findAll();
        \$spreadsheet = new Spreadsheet();
        \$sheet = \$spreadsheet->getActiveSheet();

        // Header Kolom
        \$headers = $exportHeadersString;
        \$capitalizedHeaders = array_map(fn(\$h) => ucwords(str_replace('_', ' ', \$h)), \$headers);
        \$sheet->fromArray(\$capitalizedHeaders, NULL, 'A1');

        // Data
        \$rowNumber = 2;
        foreach (\$data as \$item) {
            \$rowData = [];
            foreach (\$headers as \$header) {
                \$rowData[] = \$item[\$header] ?? '';
            }
            \$sheet->fromArray(\$rowData, NULL, 'A' . \$rowNumber++);
        }

        \$writer = new Xlsx(\$spreadsheet);
        \$filename = '$table-export-' . date('Y-m-d') . '.xlsx';

        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header("Content-Disposition: attachment; filename=\"\$filename\"");
        header('Cache-Control: max-age=0');
        \$writer->save('php://output');
        exit;
    }

    public function import()
    {
        \$permissionCheck = \$this->checkPermission('{$viewFolder}_create'); // Asumsi izin import sama dengan create
        if (\$permissionCheck instanceof \\CodeIgniter\\HTTP\\RedirectResponse) {
            return \$permissionCheck;
        }

        \$file = \$this->request->getFile('excel_file');
        if (\$file && \$file->isValid() && !\$file->hasMoved()) {
            try {
                \$spreadsheet = IOFactory::load(\$file->getTempName());
                \$sheet = \$spreadsheet->getActiveSheet();
                \$dataRows = \$sheet->toArray(null, true, true, true);

                if (count(\$dataRows) < 2) {
                    session()->setFlashdata('error', 'File Excel kosong atau hanya berisi header.');
                    return redirect()->to(base_url('/$viewFolder'));
                }

                \$headerRow = array_values(array_shift(\$dataRows));
                \$dbFields = array_map(fn(\$h) => strtolower(str_replace(' ', '_', \$h)), \$headerRow);
                
                \$importData = [];
                foreach (\$dataRows as \$row) {
                    if (empty(array_filter(array_values(\$row)))) continue; // Skip baris kosong
                    \$rowData = array_combine(\$dbFields, array_values(\$row));
                    \$importData[] = \$rowData;
                }

                if (!empty(\$importData)) {
                    \$this->model->insertBatch(\$importData);
                    session()->setFlashdata('success', count(\$importData) . ' baris data berhasil diimpor.');
                } else {
                     session()->setFlashdata('error', 'Tidak ada data valid untuk diimpor.');
                }

            } catch (\\Exception \$e) {
                session()->setFlashdata('error', 'Gagal memproses file Excel: ' . \$e->getMessage());
            }
        } else {
            session()->setFlashdata('error', 'Gagal mengunggah file. Pastikan file valid.');
        }

        return redirect()->to(base_url('/$viewFolder'));
    }

    public function template()
    {
        \$spreadsheet = new Spreadsheet();
        \$sheet = \$spreadsheet->getActiveSheet();

        // Header kolom (hanya allowedFields)
        \$headers = $templateHeadersString;
        \$capitalizedHeaders = array_map(fn(\$h) => ucwords(str_replace('_', ' ', \$h)), \$headers);
        \$sheet->fromArray(\$capitalizedHeaders, NULL, 'A1');

        \$writer = new Xlsx(\$spreadsheet);
        \$filename = '$table-template.xlsx';

        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header("Content-Disposition: attachment; filename=\"\$filename\"");
        header('Cache-Control: max-age=0');
        \$writer->save('php://output');
        exit;
    }
}
PHP;

        file_put_contents($controllerDir . $controllerName . '.php', $controllerCode);

        // === Generate Views ===
        $formFields = '';
        $tableHeaders = '';
        $tableCells = '';

        foreach ($allowedFields as $field) {
            $label = ucwords(str_replace('_', ' ', $field));
            $formFields .= <<<HTML
<div class="mb-3">
    <label for="$field" class="form-label">$label</label>
    <input type="text" class="form-control" id="$field" name="$field" value="<?= old('$field', \$item['$field'] ?? '') ?>" required>
</div>\n
HTML;
            $tableHeaders .= "<th class=\"text-capitalize\">$label</th>\n";
            $tableCells .= "<td><?= esc(\$item['$field']) ?></td>\n";
        }

        // === View index.php (Dengan Tombol Impor/Ekspor) ===
        file_put_contents("$viewDir/index.php", <<<HTML
<?= \$this->extend('layout') ?>
<?= \$this->section('content') ?>

<div class="container-fluid">
    <h1 class="mb-4">Data $singular</h1>

    <div class="d-flex flex-wrap gap-2 mb-3">
        <a href="<?= base_url('/$viewFolder/create') ?>" class="btn btn-primary">
            <i class="fas fa-plus"></i> Tambah Data
        </a>
        <a href="<?= base_url('/$viewFolder/export') ?>" class="btn btn-success">
            <i class="fas fa-file-excel"></i> Export Excel
        </a>
        <a href="<?= base_url('/$viewFolder/template') ?>" class="btn btn-outline-secondary">
             <i class="fas fa-file-download"></i> Download Template
        </a>
    </div>

    <div class="card shadow mb-4">
        <div class="card-header">
            <h6 class="m-0 font-weight-bold text-primary">Import Data dari Excel</h6>
        </div>
        <div class="card-body">
            <form action="<?= base_url('/$viewFolder/import') ?>" method="post" enctype="multipart/form-data" class="d-flex flex-wrap gap-2 align-items-center">
                <?= csrf_field() ?>
                <div class="flex-grow-1">
                    <input type="file" name="excel_file" id="excel_file" class="form-control" accept=".xlsx,.xls" required>
                </div>
                <button type="submit" class="btn btn-info">
                    <i class="fas fa-file-upload"></i> Upload & Import
                </button>
            </form>
             <small class="form-text text-muted mt-2">
                Gunakan file template yang telah diunduh untuk memastikan format kolom sesuai.
            </small>
        </div>
    </div>

    <div class="card shadow mb-4">
        <div class="card-header">
            <h6 class="m-0 font-weight-bold text-primary">Daftar Data</h6>
        </div>
        <div class="card-body">
            <div class="table-responsive">
                <table id="dataTable" class="table table-bordered table-striped" width="100%" cellspacing="0">
                    <thead><tr>$tableHeaders<th>Aksi</th></tr></thead>
                    <tbody>
                    <?php if (!empty(\$items)): ?>
                        <?php foreach (\$items as \$item): ?>
                        <tr>
                            $tableCells
                            <td>
                                <div class="d-flex gap-2">
                                    <a href="<?= base_url('/$viewFolder/show/' . \$item['id']) ?>" class="btn btn-info btn-sm"><i class="fas fa-eye"></i></a>
                                    <a href="<?= base_url('/$viewFolder/edit/' . \$item['id']) ?>" class="btn btn-warning btn-sm"><i class="fas fa-edit"></i></a>
                                    <form action="<?= base_url('/$viewFolder/delete/' . \$item['id']) ?>" method="post" class="d-inline form-delete">
                                        <?= csrf_field() ?>
                                        <input type="hidden" name="_method" value="DELETE">
                                        <button type="submit" class="btn btn-danger btn-sm btn-delete" data-name="<?= esc(\$item['id']) ?>"><i class="fas fa-trash"></i></button>
                                    </form>
                                </div>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
</div>

<?= \$this->endSection() ?>

<?= \$this->section('scripts') ?>
<script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>

<script>
document.addEventListener('DOMContentLoaded', function () {
    // Inisialisasi DataTable
    $('#dataTable').DataTable();

    // SweetAlert untuk notifikasi
    <?php if (session()->getFlashdata('success')): ?>
    Swal.fire({ icon: 'success', title: 'Berhasil', text: '<?= session()->getFlashdata('success') ?>', timer: 2500 });
    <?php elseif (session()->getFlashdata('error')): ?>
    Swal.fire({ icon: 'error', title: 'Gagal', text: '<?= session()->getFlashdata('error') ?>', timer: 4000 });
    <?php endif; ?>

    // SweetAlert untuk konfirmasi hapus
    document.querySelectorAll('.btn-delete').forEach(button => {
        button.addEventListener('click', function (e) {
            e.preventDefault();
            const form = this.closest('form');
            const name = this.getAttribute('data-name');
            Swal.fire({
                title: 'Anda Yakin?',
                text: `Data dengan ID "\${name}" akan dihapus secara permanen.`,
                icon: 'warning',
                showCancelButton: true,
                confirmButtonColor: '#d33',
                cancelButtonColor: '#3085d6',
                confirmButtonText: 'Ya, Hapus!',
                cancelButtonText: 'Batal'
            }).then((result) => {
                if (result.isConfirmed) {
                    form.submit();
                }
            });
        });
    });
});
</script>
<?= \$this->endSection() ?>
HTML);

        // === Generate view create.php ===
        file_put_contents("$viewDir/create.php", <<<HTML
<?= \$this->extend('layout') ?>
<?= \$this->section('content') ?>
<div class="container-fluid">
    <div class="card shadow">
        <div class="card-header">
            <h4>Tambah $singular</h4>
        </div>
        <div class="card-body">
            <form action="<?= base_url('/$viewFolder/store') ?>" method="post">
                <?= csrf_field() ?>
                $formFields
                <button type="submit" class="btn btn-primary">Simpan</button>
                <a href="<?= base_url('/$viewFolder') ?>" class="btn btn-secondary">Batal</a>
            </form>
        </div>
    </div>
</div>
<?= \$this->endSection() ?>
HTML);

        // === Generate view edit.php ===
        file_put_contents("$viewDir/edit.php", <<<HTML
<?= \$this->extend('layout') ?>
<?= \$this->section('content') ?>
<div class="container-fluid">
    <div class="card shadow">
         <div class="card-header">
            <h4>Edit $singular</h4>
        </div>
        <div class="card-body">
            <form action="<?= base_url('/$viewFolder/update/' . \$item['id']) ?>" method="post">
                <?= csrf_field() ?>
                <input type="hidden" name="_method" value="PUT">
                $formFields
                <button type="submit" class="btn btn-warning">Update</button>
                <a href="<?= base_url('/$viewFolder') ?>" class="btn btn-secondary">Batal</a>
            </form>
        </div>
    </div>
</div>
<?= \$this->endSection() ?>
HTML);

        // === Generate view show.php ===
        file_put_contents("$viewDir/show.php", <<<HTML
<?= \$this->extend('layout') ?>
<?= \$this->section('content') ?>
<div class="container-fluid">
     <div class="card shadow">
        <div class="card-header">
            <h4>Detail $singular</h4>
        </div>
        <div class="card-body">
            <table class="table table-bordered">
                <tbody>
                    <?php foreach (\$item as \$key => \$value): ?>
                        <tr>
                            <th class="text-capitalize" style="width: 200px;"><?= str_replace('_', ' ', esc(\$key)) ?></th>
                            <td><?= esc(\$value) ?></td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
            <a href="<?= base_url('/$viewFolder') ?>" class="btn btn-secondary mt-3">Kembali</a>
        </div>
    </div>
</div>
<?= \$this->endSection() ?>
HTML);

        session()->setFlashdata('success', "CRUD untuk tabel <strong>$table</strong> dengan fitur Impor/Ekspor berhasil dibuat di folder <strong>Output_CRUD</strong>.");
        return redirect()->to('/codegenerator');
    }

    /**
     * Auto-generate migration file jika tabel belum ada
     */
    private function createMigration($table)
    {
        $migrationName = date('YmdHis') . "_create_{$table}_table.php";
        $className = 'Create' . ucfirst($table) . 'Table';
        $migrationPath = APPPATH . 'Database/Migrations/' . $migrationName;

        $migrationCode = <<<PHP
<?php namespace App\Database\Migrations;

use CodeIgniter\Database\Migration;

class $className extends Migration
{
    public function up()
    {
        \$this->forge->addField([
            'id' => [
                'type'           => 'INT',
                'constraint'     => 11,
                'unsigned'       => true,
                'auto_increment' => true,
            ],
            // TAMBAHKAN KOLOM LAIN DI SINI SESUAI KEBUTUHAN
            // 'nama_kolom' => ['type' => 'VARCHAR', 'constraint' => '255'],
            
            'created_at' => ['type' => 'DATETIME', 'null' => true],
            'updated_at' => ['type' => 'DATETIME', 'null' => true],
        ]);
        \$this->forge->addKey('id', true);
        \$this->forge->createTable('$table');
    }

    public function down()
    {
        \$this->forge->dropTable('$table');
    }
}
PHP;

        file_put_contents($migrationPath, $migrationCode);
    }
}
