<?php

namespace App\Controllers;

use App\Models\Capaian_kinerja_prodiModel;
use App\Models\Standar_lembaga_akreditasiModel;
use App\Models\ProdiModel;
use CodeIgniter\Controller;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\IOFactory;
use App\Libraries\LogAktivitas;

class Capaian_kinerja_prodiController extends BaseController
{
    protected $Capaian_kinerja_prodiModel;
    protected $standarModel;
    protected $prodiModel;
    protected $logAktivitas;


    public function __construct()
    {
        $this->Capaian_kinerja_prodiModel = new Capaian_kinerja_prodiModel();
        $this->standarModel = new Standar_lembaga_akreditasiModel();
        $this->prodiModel   = new ProdiModel();
        $this->logAktivitas = new LogAktivitas();
    }


    public function index()
    {
        if (!session()->get('isLoggedIn')) {
            return redirect()->to('/login');
        }

        $permissionCheck = $this->checkPermission('capaian_kinerja_prodi_view');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data['items'] = $this->Capaian_kinerja_prodiModel
            ->select('capaian_kinerja_prodi.*,
                      standar_lembaga_akreditasi.nama_standar,
                      prodi.nama_prodi')
            ->join(
                'standar_lembaga_akreditasi',
                'standar_lembaga_akreditasi.id_standar = capaian_kinerja_prodi.id_standar'
            )
            ->join(
                'prodi',
                'prodi.id_prodi = capaian_kinerja_prodi.id_prodi'
            )
            ->findAll();

        return view('capaian_kinerja_prodi/index', $this->data + $data);
    }

    public function create()
    {
        $permissionCheck = $this->checkPermission('capaian_kinerja_prodi_create');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data = [
            'standar' => $this->standarModel->findAll(),
            'prodi'   => $this->prodiModel->findAll(),
        ];

        return view('capaian_kinerja_prodi/create', $this->data + $data);
    }

    public function store()
    {
        // cek permission
        $permissionCheck = $this->checkPermission('capaian_kinerja_prodi_create');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        // generate ID CAP-XXX
        $last = $this->Capaian_kinerja_prodiModel
            ->orderBy('id_capaian', 'DESC')
            ->first();

        $nextNumber = $last ? ((int) substr($last['id_capaian'], 4)) + 1 : 1;
        $idCapaian  = 'CAP-' . str_pad($nextNumber, 3, '0', STR_PAD_LEFT);

        // data yang disimpan
        $data = [
            'id_capaian'        => $idCapaian,
            'id_standar'        => $this->request->getPost('id_standar'),
            'id_prodi'          => $this->request->getPost('id_prodi'),
            'deskripsi_capaian' => $this->request->getPost('deskripsi_capaian'),
            'tahun'             => $this->request->getPost('tahun'),
            'created_by'        => session()->get('id'),
            'updated_by'        => session()->get('id'),
        ];

        // =======================
        // REKAP LOG AKTIVITAS
        // =======================
        $fields = [
            'id_capaian'        => 'ID Capaian',
            'id_standar'        => 'Standar',
            'id_prodi'          => 'Program Studi',
            'deskripsi_capaian' => 'Deskripsi Capaian',
            'tahun'             => 'Tahun',
        ];

        $keterangan = [];

        foreach ($fields as $field => $label) {
            if (!empty($data[$field])) {
                $value = esc($data[$field]);

                // khusus textarea
                if ($field === 'deskripsi_capaian') {
                    $value = nl2br($value);
                }

                $keterangan[] =
                    "<strong>{$label}</strong>:<br>" .
                    "\"{$value}\"";
            }
        }

        $keteranganLog = implode('<hr>', $keterangan);

        // simpan data
        $this->Capaian_kinerja_prodiModel->insert($data);

        // simpan log
        $user = $this->getUser();
        $this->logAktivitas->log(
            'create_capaian_prodi',
            $user['id'],
            $user['name'],
            current_url(),
            $keteranganLog
        );

        return redirect()->to('/capaian_kinerja_prodi')
            ->with('success', 'Data berhasil disimpan');
    }



    public function show($id)
    {
        $permissionCheck = $this->checkPermission('capaian_kinerja_prodi_view');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data['item'] = $this->Capaian_kinerja_prodiModel
            ->select('capaian_kinerja_prodi.*,
                      standar_lembaga_akreditasi.nama_standar,
                      prodi.nama_prodi')
            ->join(
                'standar_lembaga_akreditasi',
                'standar_lembaga_akreditasi.id_standar = capaian_kinerja_prodi.id_standar'
            )
            ->join(
                'prodi',
                'prodi.id_prodi = capaian_kinerja_prodi.id_prodi'
            )
            ->where('capaian_kinerja_prodi.id', $id)
            ->first();

        return view('capaian_kinerja_prodi/show', $data);
    }

    public function edit($id)
    {
        $permissionCheck = $this->checkPermission('capaian_kinerja_prodi_edit');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data = [
            'item'    => $this->Capaian_kinerja_prodiModel->find($id),
            'standar' => $this->standarModel->findAll(),
            'prodi'   => $this->prodiModel->findAll(),
        ];

        return view('capaian_kinerja_prodi/edit', $this->data + $data);
    }

    public function update($id)
    {
        // cek permission
        $permissionCheck = $this->checkPermission('capaian_kinerja_prodi_edit');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        // ambil data lama (sebelum update)
        $oldData = $this->Capaian_kinerja_prodiModel->find($id);

        // data baru dari form
        $newData = [
            'id_standar'        => $this->request->getPost('id_standar'),
            'id_prodi'          => $this->request->getPost('id_prodi'),
            'deskripsi_capaian' => $this->request->getPost('deskripsi_capaian'),
            'tahun'             => $this->request->getPost('tahun'),
            'updated_by'        => session()->get('id'),
        ];

        // =======================
        // REKAP PERUBAHAN LOG
        // =======================
        $fields = [
            'id_standar'        => 'Standar',
            'id_prodi'          => 'Program Studi',
            'deskripsi_capaian' => 'Deskripsi Capaian',
            'tahun'             => 'Tahun',
        ];

        $keterangan = [];

        foreach ($fields as $field => $label) {
            $oldValue = $oldData[$field] ?? '';
            $newValue = $newData[$field] ?? '';

            // hanya log jika ada perubahan
            if ($oldValue != $newValue) {
                $oldValue = esc($oldValue);
                $newValue = esc($newValue);

                if ($field === 'deskripsi_capaian') {
                    $oldValue = nl2br($oldValue);
                    $newValue = nl2br($newValue);
                }

                $keterangan[] =
                    "<strong>{$label}</strong>:<br>" .
                    "<em>Sebelum</em>: \"{$oldValue}\"<br>" .
                    "<em>Sesudah</em>: \"{$newValue}\"";
            }
        }

        // kalau tidak ada perubahan sama sekali
        $keteranganLog = !empty($keterangan)
            ? implode('<hr>', $keterangan)
            : 'Tidak ada perubahan data.';

        // update data
        $this->Capaian_kinerja_prodiModel->update($id, $newData);

        // simpan log
        $user = $this->getUser();
        $this->logAktivitas->log(
            'update_capaian_prodi',
            $user['id'],
            $user['name'],
            current_url(),
            $keteranganLog
        );

        return redirect()->to('/capaian_kinerja_prodi')
            ->with('success', 'Data berhasil diperbarui.');
    }

    public function delete($id)
    {
        // cek permission
        $permissionCheck = $this->checkPermission('capaian_kinerja_prodi_delete');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        // ambil data sebelum dihapus
        $data = $this->Capaian_kinerja_prodiModel->find($id);

        if (!$data) {
            return redirect()->to('/capaian_kinerja_prodi')
                ->with('error', 'Data tidak ditemukan.');
        }

        // =======================
        // REKAP LOG DELETE
        // =======================
        $fields = [
            'id_capaian'        => 'ID Capaian',
            'id_standar'        => 'ID Standar',
            'id_prodi'          => 'ID Prodi',
            'deskripsi_capaian' => 'Deskripsi Capaian',
            'tahun'             => 'Tahun',
        ];

        $keterangan = [];
        $keterangan[] = '<strong>Data yang dihapus:</strong>';


        foreach ($fields as $field => $label) {
            if (!empty($data[$field])) {
                $value = esc($data[$field]);

                if ($field === 'deskripsi_capaian') {
                    $value = nl2br($value);
                }

                $keterangan[] =
                    "<strong>{$label}</strong>:<br>" .
                    "\"{$value}\"";
            }
        }

        $keteranganLog = implode('<hr>', $keterangan);

        // hapus data
        $this->Capaian_kinerja_prodiModel->delete($id);

        // simpan log
        $user = $this->getUser();
        $this->logAktivitas->log(
            'delete_capaian_prodi',
            $user['id'],
            $user['name'],
            current_url(),
            $keteranganLog
        );

        return redirect()->to('/capaian_kinerja_prodi')
            ->with('success', 'Data berhasil dihapus.');
    }

    private function getUser()
    {
        return [
            'id'   => session()->get('id'),
            'name' => session()->get('name')
        ];
    }
    // --- FITUR EXCEL ---

    public function export()
    {
        $permissionCheck = $this->checkPermission('capaian_kinerja_prodi_view'); // Asumsi izin export sama dengan view
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data = $this->Capaian_kinerja_prodiModel->findAll();
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Header Kolom
        $headers = ['id', 'id_capaian', 'id_standar', 'id_prodi', 'deskripsi_capaian', 'tahun', 'created_by', 'updated_by'];
        $capitalizedHeaders = array_map(fn($h) => ucwords(str_replace('_', ' ', $h)), $headers);
        $sheet->fromArray($capitalizedHeaders, NULL, 'A1');

        // Data
        $rowNumber = 2;
        foreach ($data as $item) {
            $rowData = [];
            foreach ($headers as $header) {
                $rowData[] = $item[$header] ?? '';
            }
            $sheet->fromArray($rowData, NULL, 'A' . $rowNumber++);
        }

        $writer = new Xlsx($spreadsheet);
        $filename = 'capaian_kinerja_prodi-export-' . date('Y-m-d') . '.xlsx';

        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header("Content-Disposition: attachment; filename=\"$filename\"");
        header('Cache-Control: max-age=0');
        $writer->save('php://output');
        exit;
    }

    public function import()
    {
        $permissionCheck = $this->checkPermission('capaian_kinerja_prodi_create'); // Asumsi izin import sama dengan create
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $file = $this->request->getFile('excel_file');
        if ($file && $file->isValid() && !$file->hasMoved()) {
            try {
                $spreadsheet = IOFactory::load($file->getTempName());
                $sheet = $spreadsheet->getActiveSheet();
                $dataRows = $sheet->toArray(null, true, true, true);

                if (count($dataRows) < 2) {
                    session()->setFlashdata('error', 'File Excel kosong atau hanya berisi header.');
                    return redirect()->to(base_url('/capaian_kinerja_prodi'));
                }

                $headerRow = array_values(array_shift($dataRows));
                $dbFields = array_map(fn($h) => strtolower(str_replace(' ', '_', $h)), $headerRow);

                $importData = [];
                foreach ($dataRows as $row) {
                    if (empty(array_filter(array_values($row)))) continue; // Skip baris kosong
                    $rowData = array_combine($dbFields, array_values($row));
                    $importData[] = $rowData;
                }

                if (!empty($importData)) {
                    $this->Capaian_kinerja_prodiModel->insertBatch($importData);
                    session()->setFlashdata('success', count($importData) . ' baris data berhasil diimpor.');
                } else {
                    session()->setFlashdata('error', 'Tidak ada data valid untuk diimpor.');
                }
            } catch (\Exception $e) {
                session()->setFlashdata('error', 'Gagal memproses file Excel: ' . $e->getMessage());
            }
        } else {
            session()->setFlashdata('error', 'Gagal mengunggah file. Pastikan file valid.');
        }

        return redirect()->to(base_url('/capaian_kinerja_prodi'));
    }

    public function template()
    {
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Header kolom (hanya allowedFields)
        $headers = ['id_capaian', 'id_standar', 'id_prodi', 'deskripsi_capaian', 'tahun', 'created_by', 'updated_by'];
        $capitalizedHeaders = array_map(fn($h) => ucwords(str_replace('_', ' ', $h)), $headers);
        $sheet->fromArray($capitalizedHeaders, NULL, 'A1');

        $writer = new Xlsx($spreadsheet);
        $filename = 'capaian_kinerja_prodi-template.xlsx';

        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header("Content-Disposition: attachment; filename=\"$filename\"");
        header('Cache-Control: max-age=0');
        $writer->save('php://output');
        exit;
    }
}
