<?php

namespace App\Controllers;

use App\Models\BerkaModel;
use CodeIgniter\Controller;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\IOFactory;
use App\Libraries\LogAktivitas;

class BerkaController extends BaseController
{
    protected $BerkaModel;
    protected $logAktivitas;


    public function __construct()
    {
        $this->BerkaModel = new BerkaModel();
        $this->logAktivitas = new LogAktivitas();
    }

    public function index()
    {
        if (!session()->get('isLoggedIn')) {
            return redirect()->to('/login');
        }

        $permissionCheck = $this->checkPermission('berkas_view');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data['items'] = $this->BerkaModel->findAll();
        $user = $this->getUser();
        $this->logAktivitas->log('akses_berkas', $user['id'], $user['name'], current_url());
        return view('berkas/index', $this->data + $data);
    }

    public function create()
    {
        $permissionCheck = $this->checkPermission('berkas_create');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        return view('berkas/create', $this->data);
    }

    public function store()
    {
        $permissionCheck = $this->checkPermission('berkas_create');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $file = $this->request->getFile('file');
        if (!$file->isValid()) {
            return redirect()->back()->with('error', 'File tidak valid!');
        }

        $last = $this->BerkaModel->orderBy('id_file', 'DESC')->first();
        $nextNumber = $last ? intval(substr($last['id_file'], 5)) + 1 : 1;
        $idFile = 'FILE-' . str_pad($nextNumber, 4, '0', STR_PAD_LEFT);

        $originalName = $file->getClientName();
        $uniqueName   = $file->getRandomName();

        $file->move('uploads/berkas', $uniqueName);

        $data = [
            'id_file'       => $idFile,
            'original_name' => $originalName,
            'unique_name'   => $uniqueName,
            'id_user'       => session()->get('id'),
            'created_by'    => session()->get('id'),
            'updated_by'    => session()->get('id'),
        ];

        $this->BerkaModel->insert($data);

        $user = $this->getUser();
        $this->logAktivitas->log(
            'create_berkas',
            $user['id'],
            $user['name'],
            current_url(),
            "<strong>Menambahkan Berkas</strong><br>
         Nama File: \"{$originalName}\"<br>
         ID File: \"{$idFile}\""
        );

        return redirect()->to('/berkas')->with('success', 'Berkas berhasil diupload.');
    }



    public function edit($id_file)
    {
        $permissionCheck = $this->checkPermission('berkas_edit');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $item = $this->BerkaModel->find($id_file);

        return view('berkas/edit', $this->data + ['item' => $item]);
    }

    public function update($id_file)
    {
        $permissionCheck = $this->checkPermission('berkas_edit');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $item = $this->BerkaModel->find($id_file);
        if (!$item) {
            return redirect()->to('/berkas')->with('error', 'Data tidak ditemukan');
        }

        $file = $this->request->getFile('file');
        $perubahan = [];

        if ($file && $file->isValid()) {

            if (file_exists('uploads/berkas/' . $item['unique_name'])) {
                unlink('uploads/berkas/' . $item['unique_name']);
            }

            $uniqueName = $file->getRandomName();
            $file->move('uploads/berkas', $uniqueName);

            $update = [
                'original_name' => $file->getClientName(),
                'unique_name'   => $uniqueName,
                'updated_by'    => session()->get('id'),
            ];

            $this->BerkaModel->update($id_file, $update);

            $perubahan[] =
                "<strong>File</strong><br>" .
                "Lama: \"{$item['original_name']}\"<br>" .
                "Baru: \"{$file->getClientName()}\"";
        }

        $keterangan = empty($perubahan)
            ? 'Update tanpa perubahan file'
            : implode('<hr>', $perubahan);

        $user = $this->getUser();
        $this->logAktivitas->log(
            'update_berkas',
            $user['id'],
            $user['name'],
            current_url(),
            $keterangan
        );

        return redirect()->to('/berkas')->with('success', 'Berkas berhasil diupdate.');
    }



    public function delete($id_file)
    {
        $permissionCheck = $this->checkPermission('berkas_delete');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $item = $this->BerkaModel->find($id_file);

        if (file_exists('uploads/berkas/' . $item['unique_name'])) {
            unlink('uploads/berkas/' . $item['unique_name']);
        }

        $this->BerkaModel->delete($id_file);
        $user = $this->getUser();
        $this->logAktivitas->log('delete_berkas', $user['id'], $user['name'], current_url());
        return redirect()->to('/berkas')->with('success', 'Berkas berhasil dihapus.');
    }

    public function show($id_file)
    {
        $permissionCheck = $this->checkPermission('berkas_view');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $item = $this->BerkaModel->find($id_file);

        return view('berkas/show', $this->data + ['item' => $item]);
    }

    private function getUser()
    {
        return [
            'id'   => session()->get('id'),
            'name' => session()->get('name'),
        ];
    }
    public function view($id_file)
    {
        $item = $this->BerkaModel->find($id_file);
        if (!$item) {
            throw \CodeIgniter\Exceptions\PageNotFoundException::forPageNotFound();
        }

        $path = FCPATH . 'uploads/berkas/' . $item['unique_name'];
        if (!file_exists($path)) {
            throw \CodeIgniter\Exceptions\PageNotFoundException::forPageNotFound();
        }

        // log buka file
        $user = $this->getUser();
        $this->logAktivitas->log(
            'view_berkas',
            $user['id'],
            $user['name'],
            current_url(),
            'Membuka file: ' . $item['original_name']
        );

        return $this->response
            ->setHeader('Content-Type', mime_content_type($path))
            ->setHeader(
                'Content-Disposition',
                'inline; filename="' . $item['original_name'] . '"'
            )
            ->setBody(file_get_contents($path));
    }



    // ===============================================================
    // EXPORT DAN IMPORT (disesuaikan dengan kolom berkas yang benar)
    // ===============================================================

    public function export()
    {
        $permissionCheck = $this->checkPermission('berkas_view');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data = $this->BerkaModel->findAll();
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        $headers = ['id_file', 'original_name', 'unique_name', 'id_user', 'created_by', 'updated_by'];
        $sheet->fromArray($headers, NULL, 'A1');

        $row = 2;
        foreach ($data as $d) {
            $sheet->fromArray([
                $d['id_file'],
                $d['original_name'],
                $d['unique_name'],
                $d['id_user'],
                $d['created_by'],
                $d['updated_by']
            ], NULL, 'A' . $row++);
        }

        $writer = new Xlsx($spreadsheet);
        $filename = 'berkas-export-' . date('Y-m-d') . '.xlsx';

        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header("Content-Disposition: attachment; filename=\"$filename\"");
        header('Cache-Control: max-age=0');

        $writer->save('php://output');
        exit;
    }

    public function import()
    {
        $permissionCheck = $this->checkPermission('berkas_create'); // Asumsi izin import sama dengan create
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $file = $this->request->getFile('excel_file');
        if ($file && $file->isValid() && !$file->hasMoved()) {
            try {
                $spreadsheet = IOFactory::load($file->getTempName());
                $sheet = $spreadsheet->getActiveSheet();
                $dataRows = $sheet->toArray(null, true, true, true);

                if (count($dataRows) < 2) {
                    session()->setFlashdata('error', 'File Excel kosong atau hanya berisi header.');
                    return redirect()->to(base_url('/berkas'));
                }

                $headerRow = array_values(array_shift($dataRows));
                $dbFields = array_map(fn($h) => strtolower(str_replace(' ', '_', $h)), $headerRow);

                $importData = [];
                foreach ($dataRows as $row) {
                    if (empty(array_filter(array_values($row)))) continue; // Skip baris kosong
                    $rowData = array_combine($dbFields, array_values($row));
                    $importData[] = $rowData;
                }

                if (!empty($importData)) {
                    $this->BerkaModel->insertBatch($importData);
                    session()->setFlashdata('success', count($importData) . ' baris data berhasil diimpor.');
                } else {
                    session()->setFlashdata('error', 'Tidak ada data valid untuk diimpor.');
                }
            } catch (\Exception $e) {
                session()->setFlashdata('error', 'Gagal memproses file Excel: ' . $e->getMessage());
            }
        } else {
            session()->setFlashdata('error', 'Gagal mengunggah file. Pastikan file valid.');
        }

        return redirect()->to(base_url('/berkas'));
    }

    public function template()
    {
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Header kolom (hanya allowedFields)
        $headers = ['id_file', 'original_name', 'unique_name', 'id_user', 'created_by', 'updated_by'];
        $capitalizedHeaders = array_map(fn($h) => ucwords(str_replace('_', ' ', $h)), $headers);
        $sheet->fromArray($capitalizedHeaders, NULL, 'A1');

        $writer = new Xlsx($spreadsheet);
        $filename = 'berkas-template.xlsx';

        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header("Content-Disposition: attachment; filename=\"$filename\"");
        header('Cache-Control: max-age=0');
        $writer->save('php://output');
        exit;
    }
}
