<?php

namespace App\Controllers;

use App\Models\UserModel;
use App\Models\RoleModel;
use App\Controllers\BaseController;
use App\Libraries\LogAktivitas;

class Auth extends BaseController
{
    protected $userModel;
    protected $roleModel;
    protected $session;
    protected $logAktivitas;

    public function __construct()
    {
        helper(['form', 'url']);
        $this->userModel = new UserModel();
        $this->roleModel = new RoleModel();
        $this->session = session();
        $this->logAktivitas = new LogAktivitas();
    }

    public function login()
    {
        if ($this->session->get('isLoggedIn')) {
            return redirect()->to('/dashboard');
        }
        return view('auth/login');
    }

    public function loginPost()
    {
        $rules = [
            'email'    => 'required|valid_email',
            'password' => 'required|min_length[6]|max_length[255]',
        ];

        if (!$this->validate($rules)) {
            return view('auth/login', ['validation' => $this->validator]);
        }

        $email = $this->request->getPost('email');
        $password = $this->request->getPost('password');

        $user = $this->userModel->where('email', $email)->first();
        if (!$user) {
            // Log failed login attempt
            $this->logAktivitas->log(
                'login_failed',
                null,
                'Anonim',
                current_url(),
                'Percobaan login gagal dengan email "' . $email . '" (email tidak ditemukan)'
            );
            return view('auth/login', ['error' => 'Email tidak ditemukan']);
        }

        if (!password_verify($password, $user['password'])) {
            // Log failed login attempt
            $this->logAktivitas->log(
                'login_failed',
                $user['id'],
                $user['name'],
                current_url(),
                'Percobaan login gagal untuk user "' . $user['name'] . '" (password salah)'
            );
            return view('auth/login', ['error' => 'Password salah']);
        }

        // Ambil role dan permissions dari tabel roles
        $role = $this->roleModel->find($user['role_id']);

        $permissions = $role && !empty($role['permissions'])
            ? $role['permissions']
            : [];
        // Ensure permissions is an array (RoleModel afterFind should handle JSON decode)
        if (!is_array($permissions)) {
            $permissions = json_decode($permissions, true) ?? [];
        }

        // Set session user
        $this->session->set([
            'id'          => $user['id'],
            'name'        => $user['name'],
            'email'       => $user['email'],
            'role_id'     => $user['role_id'],
            'permissions' => $permissions,
            'isLoggedIn'  => true,
        ]);

        // Log successful login
        $this->logAktivitas->log(
            'login',
            $user['id'],
            $user['name'],
            current_url(),
            'User "' . $user['name'] . '" login ke sistem'
        );

        return redirect()->to('/dashboard');
    }

    public function logout()
    {
        // Log logout before destroying session
        $userId = $this->session->get('id');
        $userName = $this->session->get('name');
        if ($userId) {
            $this->logAktivitas->log(
                'logout',
                $userId,
                $userName,
                current_url(),
                'User "' . $userName . '" logout dari sistem'
            );
        }

        $this->session->destroy();
        return redirect()->to('/login');
    }

    public function forgotPassword()
    {
        return view('auth/forgot_password');
    }

    public function forgotPasswordPost()
    {
        $rules = [
            'email' => 'required|valid_email',
        ];

        if (!$this->validate($rules)) {
            return view('auth/forgot_password', ['validation' => $this->validator]);
        }

        $email = $this->request->getPost('email');
        $user = $this->userModel->where('email', $email)->first();

        if (!$user) {
            // Don't reveal if email exists or not for security
            return redirect()->to('/forgot-password')->with('success', 'If the email exists, a reset link has been sent.');
        }

        // Generate reset token
        $token = bin2hex(random_bytes(32));
        $expiresAt = date('Y-m-d H:i:s', strtotime('+1 hour'));

        // Update user with reset token
        $this->userModel->update($user['id'], [
            'reset_token' => $token,
            'reset_expires_at' => $expiresAt,
        ]);

        // Send email using configured settings
        $emailConfigModel = new \App\Models\EmailConfigModel();
        $emailConfig = $emailConfigModel->getActiveConfig();

        if ($emailConfig) {
            $emailService = \Config\Services::email();
            $emailService->initialize([
                'protocol' => $emailConfig['protocol'],
                'SMTPHost' => $emailConfig['smtp_host'],
                'SMTPUser' => $emailConfig['smtp_user'],
                'SMTPPass' => $emailConfig['smtp_pass'],
                'SMTPPort' => $emailConfig['smtp_port'],
                'SMTPCrypto' => $emailConfig['smtp_crypto'],
                'mailType' => 'html',
                'charset' => 'utf-8',
            ]);
            $emailService->setFrom($emailConfig['from_email'], $emailConfig['from_name']);
        } else {
            // Fallback to default config
            $emailService = \Config\Services::email();
            $emailService->setMailType('html');
        }

        $emailService->setTo($email);
        $emailService->setSubject('Password Reset Request - AUDIT TERPADU');
        $emailService->setMessage("
            <div style='font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto;'>
                <h2 style='color: #333;'>Password Reset Request</h2>
                <p>You have requested to reset your password for AUDIT TERPADU system.</p>
                <p>Please click the link below to reset your password:</p>
                <p style='margin: 20px 0;'>
                    <a href='" . base_url('reset-password/' . $token) . "' style='background-color: #007bff; color: white; padding: 10px 20px; text-decoration: none; border-radius: 5px; display: inline-block;'>Reset Password</a>
                </p>
                <p><strong>Important:</strong> This link will expire in 1 hour.</p>
                <p>If you did not request this password reset, please ignore this email.</p>
                <hr style='border: none; border-top: 1px solid #eee; margin: 20px 0;'>
                <p style='color: #666; font-size: 12px;'>This is an automated message from AUDIT TERPADU system. Please do not reply to this email.</p>
            </div>
        ");

        if ($emailService->send()) {
            return redirect()->to('/forgot-password')->with('success', 'If the email exists, a reset link has been sent.');
        } else {
            return redirect()->to('/forgot-password')->with('error', 'Failed to send email. Please try again.');
        }
    }

    public function resetPassword($token)
    {
        // Check if token is valid and not expired
        $user = $this->userModel->where('reset_token', $token)->first();

        if (!$user || strtotime($user['reset_expires_at']) < time()) {
            return redirect()->to('/login')->with('error', 'Invalid or expired reset token.');
        }

        return view('auth/reset_password', ['token' => $token]);
    }

    public function resetPasswordPost($token)
    {
        $rules = [
            'password'     => 'required|min_length[6]|max_length[255]',
            'confirm_password' => 'required|matches[password]',
        ];

        if (!$this->validate($rules)) {
            return view('auth/reset_password', ['token' => $token, 'validation' => $this->validator]);
        }

        // Check if token is valid and not expired
        $user = $this->userModel->where('reset_token', $token)->first();

        if (!$user || strtotime($user['reset_expires_at']) < time()) {
            return redirect()->to('/login')->with('error', 'Invalid or expired reset token.');
        }

        // Update password and clear reset token
        $this->userModel->update($user['id'], [
            'password' => password_hash($this->request->getPost('password'), PASSWORD_DEFAULT),
            'reset_token' => null,
            'reset_expires_at' => null,
        ]);

        // Log password reset
        $this->logAktivitas->log('password_reset', $user['id'], $user['name'], current_url());

        return redirect()->to('/login')->with('success', 'Password has been reset successfully. Please login with your new password.');
    }
}
