<?php

namespace App\Controllers;

use App\Models\AuditorModel;
use App\Models\DosenModel;
use CodeIgniter\Controller;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\IOFactory;
use App\Libraries\LogAktivitas;

class AuditorController extends BaseController
{
    protected $AuditorModel;
    protected $DosenModel;
    protected $logAktivitas;

    public function __construct()
    {
        $this->AuditorModel = new AuditorModel();
        $this->DosenModel   = new DosenModel();
        $this->logAktivitas = new LogAktivitas();
    }

    public function index()
    {
        if (!session()->get('isLoggedIn')) {
            return redirect()->to('/login');
        }

        $permissionCheck = $this->checkPermission('auditor_view');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data['items'] = $this->AuditorModel->findAll();

        return view('auditor/index', $this->data + $data);
    }

    public function create()
    {
        $permissionCheck = $this->checkPermission('auditor_create');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        // Ambil daftar dosen untuk dropdown
        $data['dosen'] = $this->DosenModel->findAll();

        return view('auditor/create', $this->data + $data);
    }

    public function store()
    {
        $permissionCheck = $this->checkPermission('auditor_create');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $post = $this->request->getPost();

        $post['user_id']    = session()->get('id');
        $post['created_by'] = session()->get('id');
        $post['updated_by'] = session()->get('id');

        $fields = [
            'id_dosen' => 'Id Dosen',
            'status_aktif' => 'Status',

        ];

        foreach ($fields as $field => $label) {
            if (!empty($post[$field])) {
                $value = esc($post[$field]);

                $keterangan[] =
                    "<strong>{$label}</strong>:<br>" .
                    "\"{$value}\"";
            }
        }

        $keteranganLog = implode('<hr>', $keterangan);
        $this->AuditorModel->insert($post);
        $user = $this->getUser();
        $this->logAktivitas->log(
            'create_auditor',
            $user['id'],
            $user['name'],
            current_url(),
            $keteranganLog
        );
        return redirect()->to('/auditor')->with('success', 'Data auditor berhasil ditambahkan.');
    }

    public function edit($id)
    {
        $permissionCheck = $this->checkPermission('auditor_edit');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $item = $this->AuditorModel->find($id);

        $data = [
            'item'  => $item,
            'dosen' => $this->DosenModel->findAll(),
        ];

        return view('auditor/edit', $this->data + $data);
    }

    public function update($id)
    {
        $permissionCheck = $this->checkPermission('auditor_edit');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $post = $this->request->getPost();
        $post['updated_by'] = session()->get('id');

        $this->AuditorModel->update($id, $post);
        $user = $this->getUser();
        $this->logAktivitas->log(
            'update_auditor',
            $user['id'],
            $user['name'],
            current_url(),
            'Update auditor baru: ' . $post['id_dosen'] .
                ' (Status: ' . $post['status_aktif'] . ')'
        );
        return redirect()->to('/auditor')->with('success', 'Data auditor berhasil diperbarui.');
    }

    public function delete($id)
    {
        $permissionCheck = $this->checkPermission('auditor_delete');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $item = $this->AuditorModel->find($id);

        if (!$item) {
            return redirect()->to('/auditor')
                ->with('error', 'Data auditor tidak ditemukan.');
        }
        $fields = [
            'id_dosen'     => 'ID Dosen',
            'status_aktif' => 'Status Aktif',
        ];

        $keterangan = [];
        $keterangan[] = '<strong>Data yang dihapus:</strong>';

        foreach ($fields as $field => $label) {
            if (!empty($item[$field])) {
                $value = esc($item[$field]);

                $keterangan[] =
                    "<strong>{$label}</strong>:<br>" .
                    "\"{$value}\"";
            }
        }

        $keteranganLog = implode('<hr>', $keterangan);

        $this->AuditorModel->delete($id);

        $user = $this->getUser();
        $this->logAktivitas->log(
            'delete_auditor',
            $user['id'],
            $user['name'],
            current_url(),
            $keteranganLog
        );

        return redirect()->to('/auditor')
            ->with('success', 'Data auditor berhasil dihapus.');
    }

    public function show($id)
    {
        $permissionCheck = $this->checkPermission('auditor_view');
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data['item'] = $this->AuditorModel->find($id);
        return view('auditor/show', $this->data + $data);
    }

    private function getUser()
    {
        return [
            'id'   => session()->get('id'),
            'name' => session()->get('name'),
        ];
    }
    // --- FITUR EXCEL ---

    public function export()
    {
        $permissionCheck = $this->checkPermission('auditor_view'); // Asumsi izin export sama dengan view
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $data = $this->AuditorModel->findAll();
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Header Kolom
        $headers = ['id', 'id_auditor', 'id_dosen', 'status_aktif', 'created_by', 'updated_by'];
        $capitalizedHeaders = array_map(fn($h) => ucwords(str_replace('_', ' ', $h)), $headers);
        $sheet->fromArray($capitalizedHeaders, NULL, 'A1');

        // Data
        $rowNumber = 2;
        foreach ($data as $item) {
            $rowData = [];
            foreach ($headers as $header) {
                $rowData[] = $item[$header] ?? '';
            }
            $sheet->fromArray($rowData, NULL, 'A' . $rowNumber++);
        }

        $writer = new Xlsx($spreadsheet);
        $filename = 'auditor-export-' . date('Y-m-d') . '.xlsx';

        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header("Content-Disposition: attachment; filename=\"$filename\"");
        header('Cache-Control: max-age=0');
        $writer->save('php://output');
        exit;
    }

    public function import()
    {
        $permissionCheck = $this->checkPermission('auditor_create'); // Asumsi izin import sama dengan create
        if ($permissionCheck instanceof \CodeIgniter\HTTP\RedirectResponse) {
            return $permissionCheck;
        }

        $file = $this->request->getFile('excel_file');
        if ($file && $file->isValid() && !$file->hasMoved()) {
            try {
                $spreadsheet = IOFactory::load($file->getTempName());
                $sheet = $spreadsheet->getActiveSheet();
                $dataRows = $sheet->toArray(null, true, true, true);

                if (count($dataRows) < 2) {
                    session()->setFlashdata('error', 'File Excel kosong atau hanya berisi header.');
                    return redirect()->to(base_url('/auditor'));
                }

                $headerRow = array_values(array_shift($dataRows));
                $dbFields = array_map(fn($h) => strtolower(str_replace(' ', '_', $h)), $headerRow);

                $importData = [];
                foreach ($dataRows as $row) {
                    if (empty(array_filter(array_values($row)))) continue; // Skip baris kosong
                    $rowData = array_combine($dbFields, array_values($row));
                    $importData[] = $rowData;
                }

                if (!empty($importData)) {
                    $this->AuditorModel->insertBatch($importData);
                    session()->setFlashdata('success', count($importData) . ' baris data berhasil diimpor.');
                } else {
                    session()->setFlashdata('error', 'Tidak ada data valid untuk diimpor.');
                }
            } catch (\Exception $e) {
                session()->setFlashdata('error', 'Gagal memproses file Excel: ' . $e->getMessage());
            }
        } else {
            session()->setFlashdata('error', 'Gagal mengunggah file. Pastikan file valid.');
        }

        return redirect()->to(base_url('/auditor'));
    }

    public function template()
    {
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Header kolom (hanya allowedFields)
        $headers = ['id_auditor', 'id_dosen', 'status_aktif', 'created_by', 'updated_by'];
        $capitalizedHeaders = array_map(fn($h) => ucwords(str_replace('_', ' ', $h)), $headers);
        $sheet->fromArray($capitalizedHeaders, NULL, 'A1');

        $writer = new Xlsx($spreadsheet);
        $filename = 'auditor-template.xlsx';

        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header("Content-Disposition: attachment; filename=\"$filename\"");
        header('Cache-Control: max-age=0');
        $writer->save('php://output');
        exit;
    }
}
